package org.kink_lang.kink.internal.str;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Converter from a sequence of bytes, to a text.
 */
public class Decoder {

    /** The backing Charset decoder. */
    private final CharsetDecoder csDecoder;

    /** The byte buffer. */
    private ByteBuffer byteBuffer;

    /** The char buffer. */
    private CharBuffer charBuffer;

    /** true after terminate() is called. */
    private boolean isTerminated = false;

    /**
     * Constructs a decoder.
     */
    Decoder(CharsetDecoder csDecoder, ByteBuffer byteBuffer, CharBuffer charBuffer) {
        this.csDecoder = csDecoder;
        this.byteBuffer = byteBuffer;
        this.charBuffer = charBuffer;
    }

    /**
     * Returns a new decoder for the charset.
     *
     * @param cs the charset.
     * @return a new decoder.
     */
    public static Decoder of(Charset cs) {
        CharsetDecoder csDecoder = cs.newDecoder()
            .onMalformedInput(CodingErrorAction.REPLACE)
            .onUnmappableCharacter(CodingErrorAction.REPLACE);
        ByteBuffer byteBuffer = ByteBuffer.allocate(100);
        CharBuffer charBuffer = CharBuffer.wrap(new char[100]);
        return new Decoder(csDecoder, byteBuffer, charBuffer);
    }

    /**
     * Terminates the decoder.
     *
     * Calling this method twice is not permitted.
     */
    public void terminate() {
        Preconds.checkState(! this.isTerminated(), "must not be terminated twice");
        this.isTerminated = true;
        this.byteBuffer.flip();

        ensureCharsCapa(this.byteBuffer.limit());
        this.csDecoder.decode(this.byteBuffer, this.charBuffer, true);
        this.csDecoder.flush(this.charBuffer);
        this.byteBuffer.compact();
    }

    /**
     * Returns true if the decoder is already terminated.
     *
     * @return true if the decoder is already terminated.
     */
    boolean isTerminated() {
        return this.isTerminated;
    }

    /**
     * Consumes bytes.
     *
     * @param bytes consumed bytes.
     */
    public void consume(byte[] bytes) {
        ensureBytesCapa(bytes.length);
        this.byteBuffer.put(bytes);
        this.byteBuffer.flip();

        ensureCharsCapa(this.byteBuffer.limit());
        this.csDecoder.decode(this.byteBuffer, this.charBuffer, false);
        this.byteBuffer.compact();
    }

    /**
     * Ensures the capacity of the byteBuffer for the added bytes.
     */
    private void ensureBytesCapa(int addedBytesLen) {
        if (addedBytesLen <= this.byteBuffer.remaining()) {
            return;
        }

        int newCapa = (this.byteBuffer.position() + addedBytesLen) * 2;
        ByteBuffer newBytes = ByteBuffer.allocate(newCapa);
        this.byteBuffer.flip();
        newBytes.put(this.byteBuffer);
        this.byteBuffer = newBytes;
    }

    /**
     * Ensures the capacity of the charBuffer for the added bytes.
     */
    private void ensureCharsCapa(int addedBytesLen) {
        int maxCharsAdded = ((int) Math.ceil(csDecoder.maxCharsPerByte())) * addedBytesLen;
        if (maxCharsAdded <= this.charBuffer.remaining()) {
            return;
        }

        int newCapa = this.charBuffer.position() + maxCharsAdded;
        CharBuffer newChars = CharBuffer.wrap(new char[newCapa]);
        this.charBuffer.flip();
        newChars.put(this.charBuffer);
        this.charBuffer = newChars;
    }

    /**
     * Emits the remaining text.
     *
     * @return the remaining text.
     */
    public String emitText() {
        this.charBuffer.flip();
        int endPos = endPosOfAvailableText(this.charBuffer);
        String result = this.charBuffer.subSequence(0, endPos).toString();
        this.charBuffer.position(endPos);
        this.charBuffer.compact();
        return result;
    }

    /**
     * Returns the end pos of the available text.
     */
    private static int endPosOfAvailableText(CharSequence chars) {
        int len = chars.length();
        return len >= 1 && Character.isHighSurrogate(chars.charAt(len - 1))
            ? len - 1
            : len;
    }

}

// vim: et sw=4 sts=4 fdm=marker
