package org.kink_lang.kink;

import java.lang.invoke.CallSite;
import java.lang.invoke.ConstantCallSite;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.invoke.MethodType;
import java.util.List;

import org.kink_lang.kink.internal.compile.classgen.ClassGenerator;
import org.kink_lang.kink.internal.compile.javaclassir.JavaClassIr;
import org.kink_lang.kink.internal.compile.javaclassir.JcirFactory;

/**
 * Bootstrap methods to create fun vals.
 */
final class FunBootstrapper {

    /**
     * Not instantiated.
     */
    private FunBootstrapper() {
    }

    /**
     * Bootstrap the callsite which makes a fun.
     *
     * @param caller the caller.
     * @param name not used.
     * @param type (Vm, [free vars... / enclosing binding]).
     * @param index the index of the fun in JavaClassIr.childJcirs.
     * @return the method handle of type (Vm, [free vars... / enclosing binding]).
     * @throws Throwable any exception.
     */
    static CallSite bootstrapMakeFun(
            Lookup caller,
            String name,
            MethodType type,
            int index) throws Throwable {
        MethodType constrMethodType = type
            .changeReturnType(void.class);
        MethodHandle mh = findConstructor(caller, index, constrMethodType);
        return new ConstantCallSite(mh.asType(type));
    }

    /**
     * Bootstrap the callsite of ()FunVal which returns a constant combinator.
     *
     * @param caller the caller.
     * @param name not used.
     * @param type ()FunVal.
     * @param index the index of the fun in JavaClassIr.childJcirs.
     * @return the method handle of ()FunVal returning the combinator.
     * @throws Throwable.
     */
    static CallSite bootstrapCombinator(
            Lookup caller,
            String name,
            MethodType type,
            int index) throws Throwable {
        MethodType constrMethodType = MethodType.methodType(void.class, Vm.class);
        MethodHandle mh = findConstructor(caller, index, constrMethodType);
        FunVal fun = (FunVal) mh.invoke(getVm(caller));
        return new ConstantCallSite(MethodHandles.constant(FunVal.class, fun));
    }

    /**
     * Finds a constructor.
     */
    private static MethodHandle findConstructor(
            Lookup caller,
            int index,
            MethodType constrMethodType) throws Throwable {
        MethodHandle childrenMh = caller
            .findStaticGetter(caller.lookupClass(), "childJcirFactories", List.class);
        List<JcirFactory> factories = (List<JcirFactory>) childrenMh.invoke();
        JavaClassIr jcir = factories.get(index).makeJcir();

        Vm vm = getVm(caller);
        ClassGenerator classGen = new ClassGenerator(vm, MethodHandles.lookup());
        Lookup generated = classGen.generate(jcir);
        return generated
            .findConstructor(generated.lookupClass(), constrMethodType);
    }

    /**
     * Gets the vm from the caller class.
     */
    private static Vm getVm(Lookup caller) throws Throwable {
        MethodHandle vmMh = caller.findStaticGetter(caller.lookupClass(), "vmStatic", Vm.class);
        return (Vm) vmMh.invoke();
    }

}

// vim: et sw=4 sts=4 fdm=marker
