package org.kink_lang.kink;

import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.OptionalInt;

import org.kink_lang.kink.hostfun.HostFunAction;
import org.kink_lang.kink.hostfun.HostFunBuilder;
import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Implementation of HostFunBuilder.
 */
class HostFunBuilderImpl implements HostFunBuilder {

    /** The vm. */
    private final Vm vm;

    /** The description of the fun. */
    private final String desc;

    /** The minimum number of args. */
    private final int argsMin;

    /** The maximum number of args. */
    private final OptionalInt argsMax;

    /**
     * Constructs a builder.
     */
    HostFunBuilderImpl(Vm vm, String desc, int argsMin, OptionalInt argsMax) {
        this.vm = vm;
        this.desc = desc;
        this.argsMin = argsMin;
        this.argsMax = argsMax;
    }

    /**
     * Returns a new builder with the specified desc,
     * and all the other properties inherited from {@code this}.
     *
     * @param desc the desc of the fun.
     * @return a new builder.
     */
    @Override
    public HostFunBuilder desc(String desc) {
        Objects.requireNonNull(desc);
        return new HostFunBuilderImpl(vm, desc, this.argsMin, this.argsMax);
    }

    /**
     * Returns a new builder with the specified number of args.
     *
     * <p>The number of args passed to a fun built by the result builder
     * must be equal to {@code numArgs}.</p>
     *
     * <p>Precondition: numArgs must be nonnegative.</p>
     *
     * @param numArgs the number of args which the fun takes.
     * @return a new builder.
     */
    @Override
    public HostFunBuilder take(int numArgs) {
        Preconds.checkArg(numArgs >= 0, "numArgs must be nonnegative");
        return new HostFunBuilderImpl(vm, this.desc, numArgs, OptionalInt.of(numArgs));
    }

    /**
     * Returns a new builder with the specified minimum number of args.
     *
     * <p>The number of args passed to a fun built by the result builder
     * must be greater than or equal to {@code argsMin}.</p>
     *
     * <p>Precondition: argsMin must be nonnegative.</p>
     *
     * @param argsMin the minimum number of args which the fun takes.
     * @retunn a new builder.
     */
    @Override
    public HostFunBuilder takeMin(int argsMin) {
        Preconds.checkArg(argsMin >= 0, "argsMin must be nonnegative");
        return new HostFunBuilderImpl(vm, this.desc, argsMin, OptionalInt.empty());
    }

    /**
     * Returns a builder with the specified minimum and the maximum number of args.
     *
     * <p>The number of args passed to a fun built by the result builder
     * must be in the range {@code [argsMin, argsMax]}.</p>
     *
     * <p>Precondition:</p>
     *
     * <ul>
     *  <li>argsMin must be nonnegative</li>
     *  <li>argsMin must be less than or equal to argsMax</li>
     * </ul>
     *
     * @param argsMin the minimum number of args which the fun takes.
     * @param argsMax the maximum number of args which the fun takes.
     * @return a new builder.
     */
    @Override
    public HostFunBuilder takeMinMax(int argsMin, int argsMax) {
        Preconds.checkArg(argsMin >= 0, "argsMin must be nonnegative");
        Preconds.checkArg(argsMin <= argsMax, "argsMin must be less than or equal to argsMax");
        return new HostFunBuilderImpl(vm, this.desc, argsMin, OptionalInt.of(argsMax));
    }

    /**
     * Makes a fun.
     *
     * @param action action which is called after args checking passes.
     * @return a fun.
     */
    @Override
    public FunVal action(HostFunAction action) {
        HostFunAction wrappedAction =
            argsMax.equals(OptionalInt.of(argsMin)) ? actionExact(desc, argsMin, action)
            : argsMax.isPresent() ? actionMinMax(desc, argsMin, argsMax.getAsInt(), action)
            : argsMin >= 1 ? actionMin(desc, argsMin, action)
            : action;
        return new HostFunVal(vm, this.desc, wrappedAction);
    }

    /**
     * Wrapping action accepting exactly {@code argsExact} args.
     */
    private static HostFunAction actionExact(String desc, int argsExact, HostFunAction rawAction) {
        return c -> {
            int args = c.argCount();
            if (args != argsExact) {
                return c.raise(String.format(Locale.ROOT,
                            "%s: required %d arg(s), but got %d arg(s)",
                            desc, argsExact, args));
            }
            return rawAction.action(c);
        };
    }

    /**
     * Wrapping action accepting {@code [argsMin, argsMax]} args.
     */
    private static HostFunAction actionMinMax(
                String desc, int argsMin, int argsMax,
                HostFunAction rawAction) {
        return c -> {
            int args = c.argCount();
            if (args < argsMin || argsMax < args) {
                return c.raise(String.format(Locale.ROOT,
                            "%s: required %d to %d arg(s), but got %d arg(s)",
                            desc, argsMin, argsMax, args));
            }
            return rawAction.action(c);
        };
    }

    /**
     * Wrapping action accepting {@code [argsMin, Inifinite]} args.
     */
    private static HostFunAction actionMin(String desc, int argsMin, HostFunAction rawAction) {
        return c -> {
            int args = c.argCount();
            if (args < argsMin) {
                return c.raise(String.format(Locale.ROOT,
                            "%s: required at least %d arg(s), but got %d arg(s)",
                            desc, argsMin, args));
            }
            return rawAction.action(c);
        };
    }

    /**
     * Returns the description of the fun.
     */
    String getDesc() {
        return this.desc;
    }

    /**
     * Returns the minimum number of args.
     */
    int getArgsMin() {
        return this.argsMin;
    }

    /**
     * Returns the maximum number of args.
     */
    OptionalInt getArgsMax() {
        return this.argsMax;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT,
                "HostFunBuilderImpl(desc=%s argsMin=%s argsMax=%s)",
                this.desc, this.argsMin, this.argsMax);
    }

    /**
     * Returns the properties of the builder.
     */
    private List<Object> getProperties() {
        return List.of(vm, desc, argsMin, argsMax);
    }

    @Override
    public int hashCode() {
        return getProperties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof HostFunBuilderImpl
            && getProperties().equals(((HostFunBuilderImpl) arg).getProperties());
    }

}

// vim: et sw=4 sts=4
