package org.kink_lang.kink.internal.callstack;

import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;

import org.kink_lang.kink.LocationVal;
import org.kink_lang.kink.TraceVal;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.contract.Preconds;

/**
 * A trace of a call.
 *
 * @param symHandle the sym handle.
 * @param location the location.
 * @param isTail whether it is on the tail.
 */
public record Trace(int symHandle, Location location, boolean isTail) {

    /** The trace which represents snipping of traces; it has neither sym handle nor location. */
    public static final Trace SNIP = new Trace(0, Location.EMPTY, true);

    /**
     * Constructs a trace.
     *
     * @param symHandle the sym handle.
     * @param location the location.
     * @param isTail whether it is on the tail.
     */
    public Trace {
        Preconds.checkArg(symHandle >= 0, "symHandle must be non-negative");
        Objects.requireNonNull(location, "location must not be null");
    }

    /**
     * Returns a trace frame with the sym handle.
     *
     * @param symHandle the sym handle.
     * @return a trace frame with the sym handle.
     */
    public static Trace of(int symHandle) {
        return produceSymOnly(symHandle, false, NON_TAIL_SYM_ONLY_TRACE_CACHE);
    }

    /**
     * Returns a trace of a taill call with the sym handle.
     *
     * @param symHandle the sym handle.
     * @return a trace of a taill call with the sym handle.
     */
    public static Trace ofTail(int symHandle) {
        return produceSymOnly(symHandle, true, TAIL_SYM_ONLY_TRACE_CACHE);
    }

    /**
     * Makes a trace with sym w/o location.
     *
     * Note that the cache reference is not used in atomic update form.
     * It can be justified because it is used as a cache,
     * and lost update is acceptable.
     */
    private static Trace produceSymOnly(
            int symHandle,
            boolean isTail,
            AtomicReference<Trace[]> cacheRef) {
        Preconds.checkArg(symHandle >= 1, "symHandle must be positive");
        Trace[] cache = cacheRef.get();
        if (symHandle >= cache.length) {
            Trace[] newCache = new Trace[symHandle + 1];
            System.arraycopy(cache, 0, newCache, 0, cache.length);
            cache = newCache;
            cacheRef.set(cache);
        }

        Trace cached = cache[symHandle];
        if (cached != null) {
            return cached;
        }

        Trace newInstance = new Trace(symHandle, Location.EMPTY, isTail);
        cache[symHandle] = newInstance;
        return newInstance;
    }

    /** Cache of traces with sym and w/o location, not on the tail. */
    private static final AtomicReference<Trace[]> NON_TAIL_SYM_ONLY_TRACE_CACHE
        = new AtomicReference<>(new Trace[] { null });

    /** Cache of traces with sym and w/o location, on the tail. */
    private static final AtomicReference<Trace[]> TAIL_SYM_ONLY_TRACE_CACHE
        = new AtomicReference<>(new Trace[] { null });

    /**
     * Returns a trace frame with the location.
     *
     * @param location the location.
     * @return a trace frame with the location.
     */
    public static Trace of(Location location) {
        return new Trace(0, location, false);
    }

    /**
     * Returns a trace frame with the sym handle and the location, not on the tail.
     *
     * @param symHandle the sym handle.
     * @param location the location.
     * @return a trace frame with the sym handle and the location, not on the tail.
     */
    public static Trace of(int symHandle, Location location) {
        return new Trace(symHandle, location, false);
    }

    /**
     * Returns a trace on tail, with the same symHandle and location.
     *
     * @return a trace on tail, with the same symHandle and location.
     */
    public Trace onTail() {
        return new Trace(this.symHandle, this.location, true);
    }

    /**
     * Returns the val representing the trace.
     *
     * @param vm the vm.
     * @return the trace val.
     */
    public TraceVal toTraceVal(Vm vm) {
        if (this.equals(SNIP)) {
            return vm.trace.snip();
        }

        TraceVal val = vm.trace.of(vm.sym.symFor(this.symHandle), produceLocVal(vm));
        return isTail() ? val.onTail() : val;
    }

    /**
     * Returns a loc val.
     */
    private LocationVal produceLocVal(Vm vm) {
        String programName = this.location.programName();
        String programText = this.location.programText();
        int pos = this.location.pos();
        return vm.location.of(programName, programText, pos);
    }

}

// vim: et sw=4 sts=4 fdm=marker
