package org.kink_lang.kink.internal.mod.random;

import java.util.Arrays;
import java.util.Random;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * The pseudo random number generator of xoshiro256**.
 *
 * See http://prng.di.unimi.it/
 */
public class Xoshiro256StarStar extends Random {

    /** The 256 bit state of the PRNG. */
    private final long[] s = new long[4];

    /** Serial version UID of the class. */
    private static final long serialVersionUID = 7026018537194797769L;

    /**
     * Constructs the PRNG with the given state.
     *
     * Precondition: state must not be all zero.
     *
     * @param s0 the first 64 bits.
     * @param s1 the second 64 bits.
     * @param s2 the third 64 bits.
     * @param s3 the fourth 64 bits.
     */
    public Xoshiro256StarStar(long s0, long s1, long s2, long s3) {
        Preconds.checkArg(s0 != 0 || s1 != 0 || s2 != 0 || s3 != 0, "state must not be all zero");
        this.s[0] = s0;
        this.s[1] = s1;
        this.s[2] = s2;
        this.s[3] = s3;
    }

    /**
     * Returns the 256 bit state of the PRNG.
     */
    long[] getState() {
        return Arrays.copyOf(this.s, this.s.length);
    }

    @Override
    public long nextLong() {
        long result = Long.rotateLeft(this.s[1] * 5, 7) * 9;
        long t = this.s[1] << 17;

        this.s[2] ^= this.s[0];
        this.s[3] ^= this.s[1];
        this.s[1] ^= this.s[2];
        this.s[0] ^= this.s[3];

        this.s[2] ^= t;
        this.s[3] = Long.rotateLeft(this.s[3], 45);

        return result;
    }

    @Override
    public boolean nextBoolean() {
        return nextLong() < 0;
    }

    @Override
    public int next(int bits) {
        return (int) (nextLong() >>> Long.SIZE - bits);
    }

    @Override
    public void nextBytes(byte[] bytes) {
        for (int i = 0; i < bytes.length; i += Long.BYTES) {
            int remainingBytes = bytes.length - i;
            long random = nextLong();
            for (int j = 0; j < Math.min(Long.BYTES, remainingBytes); ++ j) {
                int discardedBits = (Long.BYTES - (j + 1)) * Byte.SIZE;
                bytes[i + j] = (byte) (random >>> discardedBits);
            }
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
