package org.kink_lang.kink.internal.program.itree;

import java.util.HashMap;
import java.util.Map;

/**
 * Analyzes contents of local vars in ssa-funs.
 */
public final class ContentAnalysis {

    /**
     * Should not be instantiated.
     */
    private ContentAnalysis() {
    }

    /**
     * Returns the mapping from local vars to the contents.
     *
     * @param body the fun body to analyze.
     * @return the mapping from local vars to the contents.
     */
    public static Map<LocalVar, LocalVarContent> analyzeContent(Itree body) {
        Callback callback = new Callback();
        DeepTransformer.deepTransform(body, callback);
        return Map.copyOf(callback.mapping);
    }

    /**
     * Callback which does analysis.
     */
    private static class Callback
            extends SkeltonItreeVisitor<Void>
            implements DeepTransformer.Callback {

        /** Contents mapping.  */
        private final Map<LocalVar, LocalVarContent> mapping = new HashMap<>();

        /**
         * Constructs a callback.
         */
        Callback() {
            super(itree -> null);
        }

        @Override
        public LocalVar derefLvar(LocalVar lvar) {
            return lvar;
        }

        @Override
        public LocalVar storeLvar(LocalVar lvar) {
            return lvar;
        }

        @Override
        public Itree itree(Itree itree) {
            itree.accept(this);
            return itree;
        }

        @Override
        public Void visit(LstoreItree lstore) {
            Itree rhs = lstore.rhs();
            if (rhs instanceof RecvItree) {
                this.mapping.put(lstore.lvar(), new LocalVarContent.Recv());
            } else if (rhs instanceof LderefItree lderef
                    && lderef.lvar() instanceof LocalVar.Generated genLvar) {
                this.mapping.put(lstore.lvar(), new LocalVarContent.Alias(genLvar));
            }
            return null;
        }

        @Override
        public Void visit(ArgsPassingItree argsPassing) {
            for (int i = 0; i < argsPassing.lvars().size(); ++ i) {
                this.mapping.put(argsPassing.lvars().get(i), new LocalVarContent.Arg(i));
            }
            return null;
        }

        @Override
        public Void visit(NestedArgsPassingItree argsPassing) {
            for (int i = 0; i < argsPassing.params().size(); ++ i) {
                NestedParam param = argsPassing.params().get(i);
                if (param instanceof LocalVar lvar) {
                    this.mapping.put(lvar, new LocalVarContent.Arg(i));
                }
            }
            return null;
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
