package org.kink_lang.kink.internal.program.itreeoptimize;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import org.kink_lang.kink.internal.program.itree.*;

/**
 * Eliminates unused code from a seq.
 *
 * <p>Unused code are expressions (1) which do not not have side effect
 * and (2) the result of which is removed just after evaluated,
 * and (3) the execution shall not raise an exception.</p>
 */
public class EliminateUnusedOptimizer extends BaseOptimizer {

    @Override
    public Itree visit(SeqItree seq) {
        List<Itree> steps = seq.steps();
        List<Itree> stepsButLast = steps.subList(0, steps.size() - 1);
        Itree last = steps.get(steps.size() - 1);
        List<Itree> resultSteps = new ArrayList<>(stepsButLast.stream()
                .flatMap(itree -> itree.accept(new Eliminator()).stream())
                .collect(Collectors.toList()));
        resultSteps.add(last);
        return new SeqItree(resultSteps, seq.pos());
    }

    /**
     * Converts an itree to a list of itrees eliminating unused parts.
     */
    private static class Eliminator extends SkeltonItreeVisitor<List<Itree>> {

        /**
         * Constructs a visitor.
         */
        Eliminator() {
            super(itree -> List.of(itree));
        }

        @Override
        public List<Itree> visit(NadaItree itree) {
            return List.of();
        }

        @Override
        public List<Itree> visit(StrItree itree) {
            return List.of();
        }

        @Override
        public List<Itree> visit(NumItree itree) {
            return List.of();
        }

        @Override
        public List<Itree> visit(BindingItree itree) {
            return List.of();
        }

        @Override
        public List<Itree> visit(RecvItree itree) {
            return List.of();
        }

        @Override
        public List<Itree> visit(ArgVecItree itree) {
            return List.of();
        }

        @Override
        public List<Itree> visit(SlowFunItree itree) {
            return List.of();
        }

        @Override
        public List<Itree> visit(FastFunItree itree) {
            return List.of();
        }

        @Override
        public List<Itree> visit(VarrefItree itree) {
            // (A:X B) → (A B)
            return List.of(itree.owner());
        }

        @Override
        public List<Itree> visit(VecItree itree) {
            // ([A B C] D) → (A B C D)
            return itree.elems().stream().map(ItreeElem::expr).collect(Collectors.toList());
        }

        @Override
        public List<Itree> visit(LderefItree itree) {
            return itree.lvar() instanceof LocalVar.Original
                ? List.of(itree)
                : List.of();
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
