package org.kink_lang.kink.internal.program.itreeoptimize;

import java.util.ArrayList;
import java.util.List;

import org.kink_lang.kink.internal.program.itree.Itree;
import org.kink_lang.kink.internal.program.itree.ItreeElem;
import org.kink_lang.kink.internal.program.itree.StrItree;
import org.kink_lang.kink.internal.program.itree.SymValPair;
import org.kink_lang.kink.internal.program.itree.SymcallItree;
import org.kink_lang.kink.internal.program.itree.TraitNewValItree;

/**
 * Converts expression like {@code new_val(... Trait 'X' 10 'Y' 20)} to TraitNewValItree.
 */
public class TraitNewValOptimizer extends BaseOptimizer {

    @Override
    public Itree visit(SymcallItree scall) {
        if (! isOptimizable(scall)) {
            return scall;
        }

        ItreeElem trait = scall.args().get(0);
        List<SymValPair> symValPairs = new ArrayList<>();
        for (int i = 1; i < scall.args().size(); i += 2) {
            String sym = ((StrItree) scall.args().get(i).expr()).str();
            Itree val = scall.args().get(i + 1).expr();
            symValPairs.add(new SymValPair(sym, val));
        }
        return new TraitNewValItree(trait.expr(), trait.pos(), symValPairs, scall.pos());
    }

    /**
     * Whether the scall is optimizable.
     */
    private boolean isOptimizable(SymcallItree scall) {
        return scall.isLocalCallOf("new_val")
            && isOddNumOfArgs(scall.args())
            && isSpreadOnFirst(scall.args())
            && isSingleOnTail(scall.args())
            && hasSyms(scall.args());
    }

    /**
     * Whether the number of the args is odd.
     */
    private boolean isOddNumOfArgs(List<ItreeElem> args) {
        return args.size() % 2 == 1;
    }

    /**
     * Whether the first arg is spreaded.
     */
    private boolean isSpreadOnFirst(List<ItreeElem> args) {
        return args.get(0).isSpread();
    }

    /**
     * Whether the the rest args are single.
     */
    private boolean isSingleOnTail(List<ItreeElem> args) {
        return args.stream().skip(1).allMatch(ItreeElem::isSingle);
    }

    /**
     * The #odd args are strings.
     */
    private boolean hasSyms(List<ItreeElem> args) {
        for (int i = 1; i < args.size(); i += 2) {
            if (! (args.get(i).expr() instanceof StrItree)) {
                return false;
            }
        }
        return true;
    }

}

// vim: et sw=4 sts=4 fdm=marker
