package org.kink_lang.kink.internal.vec;

import java.util.Arrays;
import java.util.List;
import java.lang.invoke.VarHandle;
import java.lang.invoke.MethodHandles;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Container of vec elements.
 *
 * <p>A vec internal has a fixed capacity,
 * which is greater than or equal to the size.
 * If the size of the result of a
 * <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
 * linear update</a> method such as {@link #insert(int, Val)} is bigger than the capacity,
 * the method returns a newly created vec internal,
 * instead of modifying {@code this} vec internal.</p>
 *
 * <p>This class is weakly thread-safe,
 * in the sense that if preconditions hold,
 * actions at least return valid vals,
 * and/or don't break the state of the instance,
 * even if the instance is modified by another thread concurrently.</p>
 *
 * <p>If a read R in a thread reads a val V from the VecInternal
 * which is written by a write W in another thread,
 * any writes which happen-before W happen before
 * any reads after (in the happens-before order) R.</p>
 */
public abstract class VecInternal {

    /** The vm. */
    final Vm vm;

    /**
     * Vals containing the elems of the vec.
     * The tail is padded by nada; the elements may be modified by the subclass.
     */
    final Val[] vals;

    /** The VarHandle for VecInternal.vals. */
    static final VarHandle VALS_VH = MethodHandles.arrayElementVarHandle(Val[].class);

    /** The size of the vec; the field may be modified by the subclass. */
    int size;

    /**
     * Constructs a vec internal.
     */
    VecInternal(Vm vm, Val[] vals, int size) {
        this.vm = vm;
        this.vals = vals;
        this.size = size;
    }

    /**
     * Returns a new vec internal containing the elements of the list.
     *
     * @param vm the vm.
     * @param list the list of vals.
     * @return a new vec internal containing the elements of the list.
     */
    public static VecInternal of(Vm vm, List<? extends Val> list) {
        Val[] vals = new Val[list.size()];
        for (int i = 0; i < vals.length; ++ i) {
            vals[i] = list.get(i);
        }
        return new MutableVecInternal(vm, vals, vals.length);
    }

    /**
     * Returns a new vec containing elements between {@code from} (inclusive)
     * and {@code to} (exclusive) on the array.
     *
     * @param vm the vm.
     * @param array the array of vals.
     * @param from the beginning index (inclusive) on the array.
     * @param to the end index (exclusive) on the array.
     * @return a new vec internal.
     */
    public static VecInternal of(Vm vm, Val[] array, int from, int to) {
        Preconds.checkRange(from, to, array.length);
        Val[] vals = new Val[to - from];
        for (int i = 0; i < vals.length; ++ i) {
            vals[i] = array[from + i];
        }
        return new MutableVecInternal(vm, vals, vals.length);
    }

    /**
     * Returns a new vec internal.
     *
     * @param vm the vm.
     * @param e0 the #0 elem.
     * @return a new vec internal.
     */
    public static VecInternal of(Vm vm, Val e0) {
        return new MutableVecInternal(vm, new Val[] { e0 }, 1);
    }

    /**
     * Returns a new vec internal.
     *
     * @param vm the vm.
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @return a new vec internal.
     */
    public static VecInternal of(Vm vm, Val e0, Val e1) {
        return new MutableVecInternal(vm, new Val[] { e0, e1 }, 2);
    }

    /**
     * Returns a new vec internal.
     *
     * @param vm the vm.
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @param e2 the #2 elem.
     * @return a new vec internal.
     */
    public static VecInternal of(Vm vm, Val e0, Val e1, Val e2) {
        return new MutableVecInternal(vm, new Val[] { e0, e1, e2 }, 3);
    }

    /**
     * Returns a new vec internal.
     *
     * @param vm the vm.
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @param e2 the #2 elem.
     * @param e3 the #3 elem.
     * @return a new vec internal.
     */
    public static VecInternal of(Vm vm, Val e0, Val e1, Val e2, Val e3) {
        return new MutableVecInternal(vm, new Val[] { e0, e1, e2, e3 }, 4);
    }

    /**
     * Returns a new empty vec internal with the specified capa.
     *
     * @param vm the vm.
     * @param capa the capa.
     * @return a new empty vec internal with the specified capa.
     */
    public static VecInternal ofCapa(Vm vm, int capa) {
        Val[] array = new Val[capa];
        Arrays.fill(array, 0, capa, vm.nada);
        return new MutableVecInternal(vm, array, 0);
    }

    /**
     * Copies vals out to the dest array.
     */
    void copyOut(int from, int len, Val[] dest, int destInd) {
        for (int i = 0; i < len; ++ i) {
            Val v = (Val) VALS_VH.getAcquire(this.vals, from + i);
            dest[destInd + i] = v;
        }
    }

    /**
     * Copies vals out to the dest array, releasing.
     */
    void copyOutRelease(int from, int len, Val[] dest, int destInd) {
        for (int i = 0; i < len; ++ i) {
            Val v = (Val) VALS_VH.getAcquire(this.vals, from + i);
            VALS_VH.setRelease(dest, destInd + i, v);
        }
    }

    /**
     * Returns the size of the vec internal.
     *
     * @return the size of the vec internal.
     */
    public int size() {
        return this.size;
    }

    /**
     * Returns the indexed element.
     *
     * <p>Precondition: 0 &lt;= index &lt; capa.</p>
     *
     * @param index the index.
     * @return the indexed element.
     */
    public Val get(int index) {
        Preconds.checkElemIndex(index, capa());
        return (Val) VALS_VH.getAcquire(this.vals, index);
    }

    /**
     * Returns the trait vec internal equal to {@code this} if possible;
     * otherwise null.
     *
     * @return the trait vec internal equal to {@code this} if possible, or null.
     */
    @Nullable
    public abstract MaybeTrait getTrait();

    /**
     * Inserts a val to the end.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * @param val the inserted val.
     * @return the result vec internal; it may be {@code this}.
     */
    @CheckReturnValue
    public VecInternal append(Val val) {
        return insert(this.size, val);
    }

    /**
     * Sets the indexed element.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * @param index the index.
     * @param val the element val.
     * @return this.
     */
    @CheckReturnValue
    public abstract VecInternal set(int index, Val val);

    /**
     * Removes the indexed element from the vec internal.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= index &lt; capa.</p>
     *
     * @param index the index.
     * @return this.
     */
    @CheckReturnValue
    public abstract VecInternal remove(int index);

    /**
     * Removes the range between the indices from the vec internal.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= from &lt;= to &lt;= capa.</p>
     *
     * @param from the beginning index of the range (inclusive).
     * @param to the end index of the range (exclusive).
     * @return this.
     */
    @CheckReturnValue
    public abstract VecInternal removeRange(int from, int to);

    /**
     * Inserts a val to the position index.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= index &lt;= capa.</p>
     *
     * @param index the position index.
     * @param val the inserted val.
     * @return the result vec internal; it may be {@code this}.
     */
    @CheckReturnValue
    public abstract VecInternal insert(int index, Val val);

    /**
     * Inserts the elements of {@code added} to the position index.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= index &lt;= capa.</p>
     *
     * @param index the position index.
     * @param added the added vec internal.
     * @return the result vec internal; it may be {@code this}.
     */
    @CheckReturnValue
    public abstract VecInternal insertAll(int index, VecInternal added);

    /**
     * Inserts the elements in the range of {@code src} to the position index.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= index &lt;= capa, 0 &lt;= from &lt;= to &lt; src.length.</p>
     *
     * @param index the position index.
     * @param src array containing the added vals.
     * @param from the from index of the src.
     * @param to the to index of the src.
     * @return the result vec internal; it may be {@code this}.
     */
    @CheckReturnValue
    public abstract VecInternal insertRange(int index, Val[] src, int from, int to);

    /**
     * Returns a new vec internal copied from the range of {@code this} vec internal.
     *
     * <p>The capacity of the result vec internal is set to {@code newCapa}.</p>
     *
     * <p>Precondition: 0 &lt;= from &lt;= to &lt;= capa.</p>
     *
     * @param from the beginning index of the range (inclusive).
     * @param to the end index of the range (exclusive).
     * @param newCapa the capacity of the new vec internal.
     * @return a new vec internal.
     */
    public VecInternal copyRangeWithCapa(int from, int to, int newCapa) {
        Preconds.checkRange(from, to, capa());
        Preconds.checkArg((to - from) <= newCapa, "too small capa");
        Val[] newVals = new Val[newCapa];
        int newSize = to - from;
        copyOut(from, newSize, newVals, 0);
        Arrays.fill(newVals, newSize, newCapa, vm.nada);
        return new MutableVecInternal(vm, newVals, newSize);
    }

    /**
     * Returns a new vec internal copied from the range of {@code this} vec internal.
     *
     * <p>The capacity of the result vec internal is set to {@code to - from}.</p>
     *
     * <p>Precondition: 0 &lt;= from &lt;= to &lt;= capa.</p>
     *
     * @param from the beginning index of the range (inclusive).
     * @param to the end index of the range (exclusive).
     * @return a new vec internal.
     */
    public VecInternal copyRange(int from, int to) {
        return copyRangeWithCapa(from, to, to - from);
    }

    /**
     * Returns the new vec internal copying the whole elements of {@code this} vec internal.
     *
     * <p>The capacity of the result vec internal is set to the initial size.</p>
     *
     * @return a new vec internal.
     */
    public VecInternal copy() {
        return copyRange(0, this.size);
    }

    /**
     * Copies the vals between [0] to [size - 1]
     * to the dest array.
     *
     * @param dest the dest array.
     * @param destIndex the index of the dest array to which the vals are copied.
     * @param size the size of the range of the vals.
     */
    public void copyTo(Val[] dest, int destIndex, int size) {
        copyOut(0, size, dest, destIndex);
    }

    /**
     * Returns the capacity of the vec internal.
     *
     * @return the capacity of the vec internal.
     */
    public int capa() {
        return this.vals.length;
    }

    /**
     * Returns a list which is a view of the vec internal.
     *
     * @return a list which is a view of the vec internal.
     */
    public List<Val> asList() {
        return new VecInternalList(this);
    }

}

// vim: et sw=4 sts=4 fdm=marker
