package org.kink_lang.kink;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Container of vars which are shared among values.
 *
 * @see SharedVarsFactory#of(java.util.Map)
 *
 * @see Vm#newVal(SharedVars)
 */
public class SharedVars {

    /**
     * The ranges in this.symHandles for each hash code.
     * The upper 4 bytes is the from index (inclusive).
     * The lower 4 bytes is the to index (exclusive).
     */
    private final long[] bucketRanges;

    /** Sym handles. ORDERED BY hash(x) ASC, x ASC. */
    private final int[] symHandles;

    /** Vals for the sym handle with the same index. */
    private final Val[] vals;

    /** Id unique to each SharedVars instance. */
    private final long uniqueId;

    /** Generator of uniqueId. */
    private static final AtomicLong UNIQUE_ID_GENERATOR = new AtomicLong(0);

    /**
     * Constructs a set of shared vars.
     */
    private SharedVars(long[] bucketRanges, int[] symHandles, Val[] vals, long uniqueId) {
        this.bucketRanges = bucketRanges;
        this.symHandles = symHandles;
        this.vals = vals;
        this.uniqueId = uniqueId;
    }

    /**
     * Produces a set of shared vars.
     *
     * @param map non-empty map from sym handle to the val.
     * @return a set of shared vars.
     */
    static SharedVars of(Map<Integer, Val> map) {
        Preconds.checkArg(! map.isEmpty(), "map must not be empty");
        return build(map, map.size());
    }

    /**
     * Builds a mapping.
     */
    private static SharedVars build(Map<Integer, Val> map, int bucketCount) {
        int[] symHandles = makeSymHandles(map.keySet(), bucketCount);
        Val[] vals = makeVals(map, symHandles);
        long[] bucketRanges = makeBucketRanges(symHandles, bucketCount);
        long uniqueId = UNIQUE_ID_GENERATOR.getAndIncrement();
        return new SharedVars(bucketRanges, symHandles, vals, uniqueId);
    }

    /**
     * Makes this.symHandles.
     * ORDERED BY hash(x) ASC, x ASC.
     */
    private static int[] makeSymHandles(Set<Integer> symHandleSet, int bucketCount) {
        Comparator<Integer> compare = (x, y) -> {
            int xHash = hash(x, bucketCount);
            int yHash = hash(y, bucketCount);
            return xHash == yHash
                ? Integer.compare(x, y)
                : Integer.compare(xHash, yHash);
        };
        return symHandleSet.stream().sorted(compare).mapToInt(i -> i).toArray();
    }

    /**
     * Makes this.vals.
     */
    private static Val[] makeVals(Map<Integer, Val> map, int[] symHandles) {
        Val[] vals = new Val[symHandles.length];
        Arrays.setAll(vals, i -> map.get(symHandles[i]));
        return vals;
    }

    /**
     * The this.bucketRanges, which is the ranges in this.symHandles for each hash code.
     */
    private static long[] makeBucketRanges(int[] symHandles, int bucketCount) {
        long[] bucketRanges = new long[bucketCount];
        int prevHash = -1;
        int prevFrom = -1;
        for (int i = 0; i < symHandles.length; ++ i) {
            int hash = hash(symHandles[i], bucketCount);
            int from = hash == prevHash ? prevFrom : i;
            int to = i + 1;
            bucketRanges[hash] = makeRange(from, to);
            prevHash = hash;
            prevFrom = from;
        }
        return bucketRanges;
    }

    /**
     * Makes an element of this.bucketRanges.
     */
    private static long makeRange(int from, int to) {
        return ((long) from) << 32 | to;
    }

    /**
     * Gets the from index (inclusive) from the range.
     */
    private static int extractFromIndex(long range) {
        return (int) (range >>> 32);
    }

    /**
     * Gets the to index (exclusive) from the range.
     */
    private static int extractToIndex(long range) {
        return (int) range;
    }

    /**
     * Returns the unique id, which is unique to each SharedVars instance.
     */
    long getUniqueId() {
        return this.uniqueId;
    }

    /**
     * The index for the sym handle if any,
     * or a negative num if the shared vars does not contain the sym handle.
     *
     * The average cost of element access is O(1) because of hashing.
     *
     * The worst cost is O(log n), because it uses binarySearch for each bucket.
     */
    int getIndex(int symHandle) {
        long range = this.bucketRanges[hash(symHandle)];
        int from = extractFromIndex(range);
        int to = extractToIndex(range);
        return Arrays.binarySearch(this.symHandles, from, to, symHandle);
    }

    /**
     * Hashes the sym handle.
     */
    private int hash(int symHandle) {
        return hash(symHandle, this.bucketRanges.length);
    }

    /**
     * Hashes the sym handle with the given bucket count.
     */
    private static int hash(int symHandle, int bucketCount) {
        return Integer.remainderUnsigned(symHandle, bucketCount);
    }

    /**
     * Gets the target of the var, or null if absent.
     *
     * @param symHandle the sym handle of the var.
     * @return the target of the var, or null if absent.
     */
    @Nullable
    Val get(int symHandle) {
        int index = getIndex(symHandle);
        return index < 0 ? null : getAtIndex(index);
    }

    /**
     * Gets the target of the var at the index.
     * The result is non-null.
     */
    Val getAtIndex(int index) {
        return this.vals[index];
    }

    /**
     * Returns true if the shared vars has the var, otherwise false.
     *
     * @param symHandle the sym handle of the var.
     * @return true if the shared vars has the var, otherwise false.
     */
    boolean has(int symHandle) {
        return getIndex(symHandle) >= 0;
    }

    /**
     * Returns the set of the sym handles of the vars in this shared vars.
     *
     * @return the set of the sym handles of the vars in this shared vars.
     */
    Set<Integer> symHandleSet() {
        return Arrays.stream(this.symHandles)
            .mapToObj(i -> i)
            .collect(Collectors.toUnmodifiableSet());
    }

}

// vim: et sw=4 sts=4 fdm=marker
