package org.kink_lang.kink;

import java.util.List;
import java.util.Locale;

/**
 * A trace.
 */
public class TraceVal extends Val {

    /** The sym of the trace. */
    private final String sym;

    /** The loc of the trace. */
    private final LocationVal location;

    /** Whether the trace is on the tail. */
    private final boolean isTail;

    /**
     * Constructs a trace.
     */
    TraceVal(Vm vm, String sym, LocationVal location, boolean isTail) {
        super(vm, null);
        this.sym = sym;
        this.location = location;
        this.isTail = isTail;
    }

    /**
     * Returns a trace with all the fields of {@code this} trace and on the tail.
     *
     * @return a trace with all the fields of {@code this} trace and on the tail.
     */
    public TraceVal onTail() {
        return new TraceVal(this.vm, this.sym, this.location, true);
    }

    /**
     * Returns the sym.
     *
     * @return the sym.
     */
    public String sym() {
        return this.sym;
    }

    /**
     * Whether the sym is empty.
     */
    boolean emptySym() {
        return sym().isEmpty();
    }

    /**
     * Returns the loc.
     *
     * @return the loc.
     */
    public LocationVal location() {
        return this.location;
    }

    /**
     * Whether the loc is empty.
     */
    boolean emptyLoc() {
        return location().equals(vm.location.of("", "", 0));
    }

    /**
     * Returns whether the trace is on the tail.
     *
     * @return whether the trace is on the tail.
     */
    public boolean isTail() {
        return this.isTail;
    }

    /**
     * Returns true if the trace represents SNIP of trace frames;
     * that is, if the trace has no loc nand sym, and it is on the tail.
     *
     * @return true if the trace represents SNIP of trace frames.
     */
    public boolean isSnip() {
        return emptySym() && emptyLoc() && this.isTail;
    }

    /**
     * Returns the corresponding stack trace element to the trace.
     *
     * @return the corresponding stack trace element to the trace.
     */
    public StackTraceElement toStackTraceElement() {
        if (isSnip()) {
            return new StackTraceElement("(kink)", "(snip)", null, -1);
        }
        String methodName = emptySym() ? "(direct-call)" :  sym();
        String fileName = emptyLoc() ? null : location().programName();
        int lineNumber = emptyLoc() ? -1 : location().lineNum();
        return new StackTraceElement("(kink)", methodName, fileName, lineNumber);
    }

    @Override
    public String toString() {
        String symRepr = emptySym()
            ? "null"
            : String.format(Locale.ROOT, "«%s»", sym());
        String locRepr = emptyLoc()
            ? "null"
            : String.format(Locale.ROOT, "«%s»", location());
        return String.format(Locale.ROOT, "TraceVal(%s %s %s)", symRepr, locRepr, isTail());
    }

    /**
     * Returns the string of {@code Trace.desc}.
     *
     * @return the string of {@code Trace.desc}.
     */
    public String desc() {
        if (emptyLoc() && emptySym()) {
            return this.isTail ? "{..snip..}" : "[..empty..]";
        }

        String open = this.isTail ? "{" : "[";
        String close = this.isTail ? "}" : "]";
        if (emptyLoc()) {
            return String.format(Locale.ROOT, "%s%s%s", open, sym(), close);
        } else if (emptySym()) {
            return String.format(Locale.ROOT, "%s%s%s %s",
                    open,
                    location().desc(),
                    close,
                    location().indicator());
        } else  {
            return String.format(Locale.ROOT, "%s%s %s%s %s",
                    open,
                    location().desc(),
                    sym(),
                    close,
                    location().indicator());
        }
    }

    /**
     * Returns properties which determine equality of traces.
     */
    private List<Object> properties() {
        return List.of(vm, sym(), location(), isTail());
    }

    @Override
    public int hashCode() {
        return properties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof TraceVal argTrace
            && properties().equals(argTrace.properties());
    }

    @Override
    SharedVars sharedVars() {
        return vm.trace.sharedVars;
    }

}

// vim: et sw=4 sts=4 fdm=marker
