package org.kink_lang.kink.internal.callstack;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * A call stack.
 */
public class CallStack {

    /** The stack pointer. */
    private int sp = 0;

    /**
     * The elements.
     * 0..sp are filled by elements.
     * sp..length are null or TraceFrame.
     */
    private Cse[] elems;

    /** The trace ring buffers. */
    private TailTraceRingBuffer[] trbs;

    /** The lnums. See {@link Lnums}. */
    private long[] lnums;

    /** The maximum size which is considered moderate. */
    private final int maxModerateSize;

    /** The size of each trace ring buffer. */
    private final int trbSize;

    /**
     * Constructs a call stack.
     *
     * @param initCapa the initial capacity of the backing array.
     * @param maxModerateSize the maximum size which is considered moderate.
     * @param trbSize the max number of tail-call trace elems in a frame.
     */
    public CallStack(int initCapa, int maxModerateSize, int trbSize) {
        this.elems = new Cse[initCapa];
        this.maxModerateSize = maxModerateSize;
        this.trbSize = trbSize;
        this.trbs = makeTrbs(initCapa);
        this.lnums = new long[initCapa];
    }

    /**
     * Make initial trbs.
     */
    private TailTraceRingBuffer[] makeTrbs(int initCapa) {
        TailTraceRingBuffer[] trbs = new TailTraceRingBuffer[initCapa];
        for (int i = 0; i < initCapa; ++ i) {
            trbs[i] = makeTrb();
        }
        return trbs;
    }

    /**
     * Make a trace ring buffer.
     */
    private TailTraceRingBuffer makeTrb() {
        return TailTraceRingBuffer.withSize(this.trbSize);
    }

    /**
     * Returns true if the call stack has moderate size
     * and can push a non-trace elem.
     *
     * @return true if the call stack has moderate size.
     */
    public boolean isModerateSize() {
        return this.sp < this.maxModerateSize;
    }

    /**
     * Pushes a cse.
     *
     * <p>Precondition: {@link #isModerateSize()} must be true.</p>
     *
     * @param elem an elem.
     * @param programCounter the program counter where the processing starts if it is resumed.
     *                       The value is specific to implementation of ResumeCse.
     * @param argCount the number of args in the fram.e
     * @param dataStackUsage the size of datastack consumed by the frame.
     * @throws IllegalStateException if the elem stack has no capacity for the elem.
     */

    public void pushCse(Cse elem, int programCounter, int argCount, int dataStackUsage) {
        Preconds.checkState(isModerateSize(),
                "stack overflow; you must check it by calling isModerateSize beforehand");
        ensureCapa(this.sp + 1);
        this.elems[this.sp] = elem;
        this.trbs[this.sp].reset();
        this.lnums[this.sp] = Lnums.makeLnum(programCounter, argCount, dataStackUsage);
        ++ this.sp;
    }

    /**
     * Returns the lnum corresponding to the ResumeCse which was popped just before.
     *
     * @return the lnum corresponding to the ResumeCse which was popped just before.
     */
    public long poppedLnum() {
        return this.lnums[this.sp];
    }

    /**
     * Pushes a trace of a tail-call.
     *
     * @param trace a trace of a tail-call.
     */
    public void pushTailTrace(Trace trace) {
        this.trbs[this.sp - 1].push(trace);
    }

    /**
     * Ensures capacity of elems.
     */
    private void ensureCapa(int capa) {
        if (capa <= this.elems.length) {
            return;
        }

        int newSize = capa + 100;
        Cse[] newFrames = new Cse[newSize];
        System.arraycopy(this.elems, 0, newFrames, 0, this.elems.length);
        this.elems = newFrames;

        TailTraceRingBuffer[] newTrbs = new TailTraceRingBuffer[newSize];
        System.arraycopy(this.trbs, 0, newTrbs, 0, this.trbs.length);
        for (int i = this.trbs.length; i < newSize; ++ i) {
            newTrbs[i] = makeTrb();
        }
        this.trbs = newTrbs;

        long[] newLnums = new long[newSize];
        System.arraycopy(this.lnums, 0, newLnums, 0, this.lnums.length);
        this.lnums = newLnums;
    }

    /**
     * Removes elems till the topmost {@link ResumeCse},
     * and returns the elem.
     *
     * @return the topmost {@link ResumeCse}.
     * @throws IllegalStateException if there is no {@link ResumeCse}.
     */
    public ResumeCse popResumer() {
        // search a consumer from top to bottom
        while (this.sp >= 1) {
            Cse elem = removeTop();
            if (elem instanceof ResumeCse) {
                return (ResumeCse) elem;
            }
        }

        throw new IllegalStateException("no ResumeCse");
    }

    /**
     * Removes the {@link FakeCallTraceCse} on the top.
     *
     * @throws IllegalStateException if the top is not {@link FakeCallTraceCse}.
     */
    public void popFakeCallTrace() {
        Cse top = removeTop();
        Preconds.checkState(top instanceof FakeCallTraceCse,"top must be fake call trace");
    }

    /**
     * Removes the top elem.
     */
    private Cse removeTop() {
        Cse top = top();
        this.elems[sp - 1] = null;
        -- this.sp;
        return top;
    }

    /**
     * Returns the top elem.
     */
    private Cse top() {
        return this.elems[sp - 1];
    }

    /**
     * Returns the index of the nearest kont tag.
     */
    private int findDelimiterIndex(KontTagCse target) {
        for (int i = sp - 1; i >= 0; -- i) {
            Cse elem = this.elems[i];
            if (elem.equals(target)) {
                return i;
            }
        }
        return -1;
    }

    /**
     * Returns true when the elem stack contains at least one kontTag.
     *
     * @param kontTag the kont tag.
     * @return true the call stack contains at least one kontTag.
     */
    public boolean canAbort(KontTagCse kontTag) {
        int index = findDelimiterIndex(kontTag);
        return index >= 0;
    }

    /**
     * Performs abort operation on shift function.
     *
     * <p>This method returns the elems from the nearest {@code kontTag}
     * to the current stack top.
     * This method removes the elems after the ResetFrame.</p>
     *
     * <p>Precondition: {@link #canAbort(KontTagCse)} must return true.</p>
     *
     * @param kontTag the kont tag.
     * @return the delimited continuation elems.
     *
     * @throws IllegalArgumentException
     *      if the call stack contains no {@code kontTag}.
     */
    public CallStackSlice abort(KontTagCse kontTag) {
        int delimInd = findDelimiterIndex(kontTag);
        Preconds.checkArg(delimInd >= 0, "kontTag must exist on the call stack");
        int contEnd = this.sp;
        int contSize = contEnd - delimInd;
        Cse[] contCses = extractContCses(delimInd, contSize);
        Arrays.fill(this.elems, delimInd + 1, this.sp, null);

        TailTraceRingBuffer[] contTrbs = extractContTrbs(delimInd, contSize);
        this.trbs[delimInd].reset();

        long[] contLnums = extractContLnums(delimInd, contSize);

        this.sp = delimInd + 1;

        return new CallStackSlice(contCses, contTrbs, contLnums);
    }

    /**
     * Extract cont cses.
     */
    private Cse[] extractContCses(int delimInd, int contSize) {
        Cse[] slice = new Cse[contSize];
        System.arraycopy(this.elems, delimInd, slice, 0, contSize);
        return slice;
    }

    /**
     * Extract cont trbs.
     */
    private TailTraceRingBuffer[] extractContTrbs(int delimInd, int contSize) {
        TailTraceRingBuffer[] slice = new TailTraceRingBuffer[contSize];
        for (int i = 0; i < contSize; ++ i) {
            slice[i] = this.trbs[delimInd + i].makeCopy();
        }
        return slice;
    }

    /**
     * Gets lnums back to the delimitor.
     */
    private long[] extractContLnums(int delimInd, int contSize) {
        long[] contLnums = new long[contSize];
        System.arraycopy(this.lnums, delimInd, contLnums, 0, contSize);
        return contLnums;
    }

    /**
     * Returns true if the call stack can replay the slice.
     *
     * @param callSlice the delimited continuation slice.
     * @return true if the call stack can replay the cont elems.
     */
    public boolean canReplay(CallStackSlice callSlice) {
        return this.sp + callSlice.size() <= this.maxModerateSize;
    }

    /**
     * Replays the delimited continuation.
     *
     * <p>Precondition: {@link #canReplay(CallStackSlice)} must return true.
     * {@code contFrames} must be a result of {@link #abort(KontTagCse)}</p>
     *
     * @param slice the delimited continuation slice.
     */
    public void replay(CallStackSlice slice) {
        Preconds.checkState(canReplay(slice),
                "stack overflow; must check by canReplay beforehand");
        ensureCapa(this.sp + slice.size());

        // replay CSEs
        System.arraycopy(slice.callStackElements(), 0, this.elems, this.sp, slice.size());

        // replay tail traces
        for (int i = 0; i < slice.size(); ++ i) {
            TailTraceRingBuffer trb = this.trbs[this.sp + i];
            trb.copyFrom(slice.tailTraceRingBuffers()[i]);
        }

        // replay lnums
        System.arraycopy(slice.lnums(), 0, this.lnums, this.sp, slice.size());

        this.sp += slice.size();
    }

    /**
     * Returns the call stack elems.
     */
    List<Cse> elems() {
        return List.copyOf(Arrays.asList(elems).subList(0, sp));
    }

    /**
     * Returns the traces; bottom to top.
     *
     * @return the traces; bottom to top.
     */
    public List<Trace> traces() {
        List<Trace> traces = new ArrayList<>();
        for (int i = 0; i < this.sp; ++ i) {
            int programCounter = Lnums.getProgramCounter(this.lnums[i]);
            Cse cse = this.elems[i];
            traces.add(cse.trace(programCounter));
            traces.addAll(this.trbs[i].traces());
        }
        return Collections.unmodifiableList(traces);
    }

    /**
     * Returns the lnums; bottom to top.
     */
    long[] lnums() {
        return Arrays.copyOf(this.lnums, this.sp);
    }

}

// vim: et sw=4 sts=4 fdm=marker
