package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.objectweb.asm.Type;
import org.objectweb.asm.commons.Method;

import org.kink_lang.kink.BindingVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.program.itree.*;

/**
 * Compiler of slow funs.
 */
public class SlowFunCompiler {

    /** The vm. */
    private final Vm vm;

    /** The program name. */
    private final String programName;

    /** The program text. */
    private final String programText;

    /** Logger for the class. */
    private static final System.Logger LOGGER
        = System.getLogger(SlowFunCompiler.class.getName());

    /** Insn to invoke BindingVal.dup. */
    private static final Insn INVOKE_DUP = new Insn.InvokeVirtual(
            Type.getType(BindingVal.class),
            new Method("dup", Type.getType(BindingVal.class), new Type[0]));

    /**
     * Constructs a compiler.
     *
     * @param vm the vm.
     * @param programName the program name.
     * @param programText the program text.
     */
    public SlowFunCompiler(Vm vm, String programName, String programText) {
        this.vm = vm;
        this.programName = programName;
        this.programText = programText;
    }

    /**
     * Compiles a slow fun into JavaClassIr.
     *
     * @param fun the fun to compile.
     * @return the compiled JavaClassIr.
     */
    public JavaClassIr compile(SlowFunItree fun) {
        var keySup = new KeyStrSupplier();
        var traceAccum = new TraceAccumulator();
        var bindingGen = BindingGenerator.STACK;
        var lvarAccGen = new SlowLvarAccessGenerator(vm, bindingGen, keySup, traceAccum);
        var nonCombinatorCompiler = new BindingCaptureFastFunCompiler(vm, programName, programText);
        var combinatorCompiler = new ValCaptureFastFunCompiler(vm, programName, programText);
        var jcirAccum = new ChildJcirAccumulator();
        var makeFastFunGen = new MakeBindingCaptureFastFunGenerator(
                nonCombinatorCompiler::compile,
                combinatorCompiler::compileControlUnchanged,
                bindingGen,
                jcirAccum);
        var controlGen = new OverriddenControlGenerator(
                vm, programName, programText, keySup, traceAccum);
        InsnsGenerator insnsGen = new InsnsGenerator(vm, programName, programText,
                bindingGen,
                lvarAccGen,
                makeFastFunGen,
                new InSlowFunLetRecGenerator(),
                controlGen,
                keySup,
                traceAccum,
                new ProgramCounterSupplier(traceAccum),
                jcirAccum);

        List<Insn> insns = new ArrayList<>(CompilerSupport.PROLOGUE);

        // dataStack.push(((BindingVal) this.valField0).dup())
        insns.add(InsnsGenerator.LOAD_DATASTACK);
        insns.add(new Insn.LoadThis());
        insns.add(new Insn.GetField(JavaClassIr.TYPE_BASE, "valField0", Type.getType(Val.class)));
        insns.add(new Insn.CheckCast(Type.getType(BindingVal.class)));
        insns.add(INVOKE_DUP);
        insns.add(InsnsGenerator.INVOKE_PUSH_TO_DATASTACK);

        insns.addAll(insnsGen.generate(fun.body(), ResultContext.TAIL));
        insns.addAll(CompilerSupport.EPILOGUE);
        String desc = String.format(Locale.ROOT, "(slow-fun location=%s)",
                vm.location.of(programName, programText, fun.pos()).desc());
        LOGGER.log(System.Logger.Level.TRACE, "insns for {0}: {1}", desc, insns);
        return new JavaClassIr(
                1,
                insns,
                traceAccum.traces(),
                desc,
                jcirAccum.childJcirFactories());
    }

}

// vim: et sw=4 sts=4 fdm=marker
