package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.kink_lang.kink.internal.program.itree.FastFunItree;

import org.kink_lang.kink.Vm;

/**
 * Compiles val-capture fast funs.
 */
public class ValCaptureFastFunCompiler {

    /** Logger for the class. */
    private static final System.Logger LOGGER
        = System.getLogger(BindingCaptureFastFunCompiler.class.getName());

    /** The vm. */
    private final Vm vm;

    /** The name of the program. */
    private final String programName;

    /** The text of the program. */
    private final String programText;

    /**
     * Constructs a compiler.
     *
     * @param vm the vm.
     * @param programName the name of the program.
     * @param programText the text of the program.
     */
    public ValCaptureFastFunCompiler(Vm vm, String programName, String programText) {
        this.vm = vm;
        this.programName = programName;
        this.programText = programText;
    }

    /**
     * Copmiles an val-capture fast fun for the case that local vars of control funs
     * are unchanged on the top level.
     *
     * @param fun the fun to compile.
     * @return the compiled IR.
     */
    public JavaClassIr compileControlUnchanged(FastFunItree fun) {
        var allocationSet = AllocationSet.valCaptureControlUnchanged(fun);
        var keySup = new KeyStrSupplier();
        var traceAccum = new TraceAccumulator();
        var lvarAccGen = new FastLvarAccessGenerator(allocationSet, keySup, traceAccum);
        var jcirAccum = new ChildJcirAccumulator();
        var controlGen = new UnchangedControlGenerator(
                vm, programName, programText, keySup, traceAccum);
        var makeFunGen = new MakeValCaptureFastFunGenerator(
                lvarAccGen,
                this::compileControlUnchanged,
                AllocationSet::valCaptureControlUnchanged,
                jcirAccum);
        return compile(
                fun,
                allocationSet,
                keySup,
                traceAccum,
                lvarAccGen,
                jcirAccum,
                controlGen,
                makeFunGen,
                "unchanged");
    }

    /**
     * Copmiles an val-capture fast fun for the case that a local var of a control fun
     * is overridden on the top level.
     *
     * @param fun the fun to compile.
     * @return the compiled IR.
     */
    public JavaClassIr compileControlOverridden(FastFunItree fun) {
        var allocationSet = AllocationSet.valCaptureControlOverridden(fun);
        var keySup = new KeyStrSupplier();
        var traceAccum = new TraceAccumulator();
        var lvarAccGen = new FastLvarAccessGenerator(allocationSet, keySup, traceAccum);
        var jcirAccum = new ChildJcirAccumulator();
        var controlGen = new OverriddenControlGenerator(
                vm, programName, programText, keySup, traceAccum);
        var makeFunGen = new MakeValCaptureFastFunGenerator(
                lvarAccGen,
                this::compileControlOverridden,
                AllocationSet::valCaptureControlOverridden,
                jcirAccum);
        return compile(
                fun,
                allocationSet,
                keySup,
                traceAccum,
                lvarAccGen,
                jcirAccum,
                controlGen,
                makeFunGen,
                "overridden");
    }

    /**
     * Generates JavaClassIr.
     */
    private JavaClassIr compile(
            FastFunItree fun,
            AllocationSet allocationSet,
            KeyStrSupplier keySup,
            TraceAccumulator traceAccum,
            LvarAccessGenerator lvarAccGen,
            ChildJcirAccumulator jcirAccum,
            ControlGenerator controlGen,
            MakeFastFunGenerator makeFunGen,
            String controlDesc) {
        var letRecGen = new InFastFunLetRecGenerator(lvarAccGen, makeFunGen);
        var pcSup = new ProgramCounterSupplier(traceAccum);
        InsnsGenerator insnsGen = new InsnsGenerator(
                vm, programName, programText,
                BindingGenerator.NOT_AVAILABLE,
                lvarAccGen,
                makeFunGen,
                letRecGen,
                controlGen,
                keySup,
                traceAccum,
                pcSup,
                jcirAccum);
        List<Insn> insns = doResume(fun, allocationSet, insnsGen);
        String desc = String.format(Locale.ROOT,
                "(val-capture-fast-fun location=(%s) control=%s)",
                vm.location.of(programName, programText, fun.pos()).desc(), controlDesc);
        LOGGER.log(System.Logger.Level.TRACE, "insns for {0}: {1}", desc, insns);
        return new JavaClassIr(
                allocationSet.field().size(),
                insns,
                traceAccum.traces(),
                desc,
                jcirAccum.childJcirFactories());
    }

    /**
     * Makes insns of doResume.
     */
    private List<Insn> doResume(
            FastFunItree fun, AllocationSet allocationSet, InsnsGenerator insnsGen) {
        List<Insn> insns = new ArrayList<>(CompilerSupport.PROLOGUE);

        // allocate lvars
        insns.addAll(CompilerSupport.allocateStack(allocationSet.stack().size()));

        // fun body
        insns.addAll(insnsGen.generate(fun.body(), ResultContext.TAIL));

        insns.addAll(CompilerSupport.EPILOGUE);
        return insns;
    }

}

// vim: et sw=4 sts=4 fdm=marker
