package org.kink_lang.kink.internal.mod.java;

import java.lang.invoke.MethodHandles;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Locale;

import org.kink_lang.kink.*;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * Factory of JAVA mod.
 */
public class JavaMod {

    /** The vm. */
    private final Vm vm;

    /** The name of the mod. */
    public static final String MOD_NAME = "kink/javahost/JAVA";

    /**
     * Constructs the factory.
     */
    private JavaMod(Vm vm) {
        this.vm = vm;
    }

    /**
     * Returns {@code JAVA} mod for the vm.
     *
     * @param vm the vm.
     * @return the JAVA mod.
     */
    public static Val makeMod(Vm vm) {
        return new JavaMod(vm).makeMod();
    }

    /**
     * Makes JAVA mod.
     */
    private Val makeMod() {
        Val javaMod = vm.newVal();
        javaMod.setVar(vm.sym.handleFor("is?"),
                vm.fun.make("JAVA.is?(Val)")
                .take(1).action(c -> vm.bool.of(c.arg(0) instanceof JavaVal)));
        javaMod.setVar(vm.sym.handleFor("string"),
                vm.fun.make("JAVA.string(Str)").take(1).action(this::stringFun));
        javaMod.setVar(vm.sym.handleFor("bytes"),
                vm.fun.make("JAVA.bytes(Bin)").take(1).action(this::bytesFun));
        javaMod.setVar(vm.sym.handleFor("boolean"),
                vm.fun.make("JAVA.boolean(Bool)").take(1).action(this::booleanFun));
        javaMod.setVar(vm.sym.handleFor("char"),
                vm.fun.make("JAVA.char(Num)").take(1).action(this::charFun));
        javaMod.setVar(vm.sym.handleFor("byte"),
                vm.fun.make("JAVA.byte(Num)").take(1).action(this::byteFun));
        javaMod.setVar(vm.sym.handleFor("short"),
                vm.fun.make("JAVA.short(Num)").take(1).action(this::shortFun));
        javaMod.setVar(vm.sym.handleFor("int"),
                vm.fun.make("JAVA.int(Num)").take(1).action(this::intFun));
        javaMod.setVar(vm.sym.handleFor("long"),
                vm.fun.make("JAVA.long(Num)").take(1).action(this::longFun));
        javaMod.setVar(vm.sym.handleFor("float"),
                vm.fun.make("JAVA.float(Num)").take(1).action(this::floatFun));
        javaMod.setVar(vm.sym.handleFor("double"),
                vm.fun.make("JAVA.double(Num)").take(1).action(this::doubleFun));
        javaMod.setVar(vm.sym.handleFor("big_integer"),
                vm.fun.make("JAVA.big_integer(Num)").take(1).action(this::bigIntegerFun));
        javaMod.setVar(vm.sym.handleFor("big_decimal"),
                vm.fun.make("JAVA.big_decimal(Num)").take(1).action(this::bigDecimalFun));
        javaMod.setVar(vm.sym.handleFor("wrap"),
                vm.fun.make("JAVA.wrap").take(1).action(this::wrapFun));
        javaMod.setVar(vm.sym.handleFor("boolean_class"),
                vm.fun.constant(vm.java.of(boolean.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("char_class"),
                vm.fun.constant(vm.java.of(char.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("byte_class"),
                vm.fun.constant(vm.java.of(byte.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("short_class"),
                vm.fun.constant(vm.java.of(short.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("int_class"),
                vm.fun.constant(vm.java.of(int.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("long_class"),
                vm.fun.constant(vm.java.of(long.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("float_class"),
                vm.fun.constant(vm.java.of(float.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("double_class"),
                vm.fun.constant(vm.java.of(double.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("void_class"),
                vm.fun.constant(vm.java.of(void.class, Class.class)));
        javaMod.setVar(vm.sym.handleFor("null"),
                vm.fun.constant(vm.java.of(null, Object.class)));
        javaMod.setVar(vm.sym.handleFor("true"),
                vm.fun.constant(vm.java.of(true, boolean.class)));
        javaMod.setVar(vm.sym.handleFor("false"),
                vm.fun.constant(vm.java.of(false, boolean.class)));
        javaMod.setVar(vm.sym.handleFor("class"),
                vm.fun.make("JAVA.class(Class_name)").take(1).action(this::classFun));
        return javaMod;
    }

    /**
     * Implementation of JAVA.string.
     */
    private HostResult stringFun(CallContext c) {
        Val strVal = c.arg(0);
        if (! (strVal instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.string(Str): Str must be a str, but got {}",
                        vm.graph.repr(strVal)));
        }
        return vm.java.of(((StrVal) strVal).string(), String.class);
    }

    /**
     * Implementation of JAVA.bytes(Bin).
     */
    private HostResult bytesFun(CallContext c) {
        Val binVal = c.arg(0);
        if (! (binVal instanceof BinVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.bytes(Bin): Bin must be a bin, but got {}",
                        vm.graph.repr(binVal)));
        }
        return vm.java.of(((BinVal) binVal).bytes(), byte[].class);
    }

    /**
     * Implementation of JAVA.boolean(Bool).
     */
    private HostResult booleanFun(CallContext c) {
        Val boolVal = c.arg(0);
        if (boolVal == vm.bool.trueVal) {
            return vm.java.of(true, boolean.class);
        } else if (boolVal == vm.bool.falseVal) {
            return vm.java.of(false, boolean.class);
        } else {
            return c.call(vm.graph.raiseFormat(
                        "JAVA.boolean(Bool): Bool must be a bool, but got {}",
                        vm.graph.repr(boolVal)));
        }
    }

    /**
     * Implementation of JAVA.char(Num).
     */
    private HostResult charFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.char(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        int num = ((NumVal) numVal).bigDecimal().intValue();
        return vm.java.of((char) num, char.class);
    }

    /**
     * Implementation of JAVA.byte.
     */
    private HostResult byteFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.byte(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        return vm.java.of(((NumVal) numVal).bigDecimal().byteValue(), byte.class);
    }

    /**
     * Implementation of JAVA.short.
     */
    private HostResult shortFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.short(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        return vm.java.of(((NumVal) numVal).bigDecimal().shortValue(), short.class);
    }

    /**
     * Implementation of JAVA.int.
     */
    private HostResult intFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.int(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        return vm.java.of(((NumVal) numVal).bigDecimal().intValue(), int.class);
    }

    /**
     * Implementation of JAVA.long.
     */
    private HostResult longFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.long(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        return vm.java.of(((NumVal) numVal).bigDecimal().longValue(), long.class);
    }

    /**
     * Implementation of JAVA.float.
     */
    private HostResult floatFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.float(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        return vm.java.of(((NumVal) numVal).bigDecimal().floatValue(), float.class);
    }

    /**
     * Implementation of JAVA.double.
     */
    private HostResult doubleFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat("JAVA.double(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        return vm.java.of(((NumVal) numVal).bigDecimal().doubleValue(), double.class);
    }

    /**
     * Implementation of JAVA.big_integer.
     */
    private HostResult bigIntegerFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat(
                        "JAVA.big_integer(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        return vm.java.of(((NumVal) numVal).bigDecimal().toBigInteger(), BigInteger.class);
    }

    /**
     * Implementation of JAVA.big_decimal.
     */
    private HostResult bigDecimalFun(CallContext c) {
        Val numVal = c.arg(0);
        if (! (numVal instanceof NumVal)) {
            return c.call(vm.graph.raiseFormat(
                        "JAVA.big_decimal(Num): Num must be a num, but got {}",
                        vm.graph.repr(numVal)));
        }
        return vm.java.of(((NumVal) numVal).bigDecimal(), BigDecimal.class);
    }

    /**
     * Implementation of JAVA.wrap(Val).
     */
    private HostResult wrapFun(CallContext c) {
        return vm.java.of(c.arg(0), Val.class);
    }

    /**
     * Implementation of JAVA.class(Name).
     */
    private HostResult classFun(CallContext c) {
        Val nameVal = c.arg(0);
        if (! (nameVal instanceof StrVal nameStr)) {
            return c.call(vm.graph.raiseFormat(
                        "JAVA.class(Class_name): Class_name must be a str, but got {}",
                        vm.graph.repr(nameVal)));
        }
        String name = nameStr.string();
        var lookup = MethodHandles.lookup()
            .dropLookupMode(MethodHandles.Lookup.MODULE);
        try {
            Class<?> klass = lookup.findClass(name);
            return vm.java.of(klass, Class.class);
        } catch (IllegalAccessException iae) {
            return c.raise(String.format(Locale.ROOT,
                        "JAVA.class(Class_name): no access privilege to %s", name));
        } catch (ClassNotFoundException cnfe) {
            return c.raise(String.format(Locale.ROOT,
                        "JAVA.class(Class_name): class not found %s", name));
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
