package org.kink_lang.kink.internal.vec;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;

import org.kink_lang.kink.StrVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Ordinary vec internal whose contents are modifiable.
 */
class MutableVecInternal extends VecInternal {

    /**
     * Constructs a vec internal.
     */
    MutableVecInternal(Vm vm, Val[] vals, int size) {
        super(vm, vals, size);
    }

    /**
     * Slides the range to left.
     */
    private void slideLeft(int src, int dest, int len) {
        for (int i = 0; i < len; ++ i) {
            Val v = (Val) VALS_VH.getAcquire(this.vals, src + i);
            VALS_VH.setRelease(this.vals, dest + i, v);
        }
    }

    /**
     * Slides the range to right.
     */
    private void slideRight(int src, int dest, int len) {
        for (int i = len - 1; i >= 0; -- i) {
            Val v = (Val) VALS_VH.getAcquire(this.vals, src + i);
            VALS_VH.setRelease(this.vals, dest + i, v);
        }
    }

    /**
     * Fills nada into the range.
     */
    private void fillNada(int from, int to) {
        for (int i = from; i < to; ++ i) {
            VALS_VH.setRelease(this.vals, i, vm.nada);
        }
    }

    /**
     * Copies vals from the src array.
     */
    private void copyIn(Val[] src, int from, int len, int destInd) {
        for (int i = 0; i < len; ++ i) {
            Val v = src[from + i];
            VALS_VH.setRelease(this.vals, destInd + i, v);
        }
    }

    @Override
    @Nullable
    public MaybeTrait getTrait() {
        int size = this.size;
        if (size % 2 != 0) {
            return new TraitError.ArityNotEven(size);
        }

        Val[] newVals = new Val[size];
        Map<Integer, Val> mapping = new HashMap<>();
        for (int i = 0; i < size; i += 2) {
            Val symVal = (Val) VALS_VH.getAcquire(this.vals, i);
            if (! (symVal instanceof StrVal sym)) {
                return new TraitError.SymNotStr(i, symVal);
            }

            Val content = (Val) VALS_VH.getAcquire(this.vals, i + 1);
            newVals[i] = sym;
            newVals[i + 1] = content;
            int symHandle = vm.sym.handleFor(sym.string());
            mapping.put(symHandle, content);
        }
        return new TraitVecInternal(vm, newVals, vm.sharedVars.of(mapping));
    }

    /**
     * Sets the indexed element.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * @param index the index.
     * @param val the element val.
     * @return this.
     */
    @Override
    @CheckReturnValue
    public VecInternal set(int index, Val val) {
        Preconds.checkElemIndex(index, capa());
        VALS_VH.setRelease(this.vals, index, val);
        return this;
    }

    /**
     * Removes the indexed element from the vec internal.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= index &lt; capa.</p>
     *
     * @param index the index.
     * @return this.
     */
    @Override
    @CheckReturnValue
    public VecInternal remove(int index) {
        Preconds.checkElemIndex(index, capa());
        int orgSize = this.size;
        int newSize = orgSize - 1;
        int moved = newSize - index;
        if (moved < 0) {
            return this;
        }
        slideLeft(index + 1, index, moved);
        VALS_VH.setRelease(this.vals, newSize, vm.nada);
        this.size = newSize;
        return this;
    }

    /**
     * Removes the range between the indices from the vec internal.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= from &lt;= to &lt;= capa.</p>
     *
     * @param from the beginning index of the range (inclusive).
     * @param to the end index of the range (exclusive).
     * @return this.
     */
    @Override
    @CheckReturnValue
    public VecInternal removeRange(int from, int to) {
        Preconds.checkRange(from, to, capa());
        int orgSize = this.size;
        int newSize = orgSize - (to - from);
        int moved = orgSize - to;
        if (moved < 0) {
            return this;
        }
        slideLeft(to, from, moved);
        fillNada(newSize, orgSize);
        this.size = newSize;
        return this;
    }

    /**
     * Inserts a val to the position index.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= index &lt;= capa.</p>
     *
     * @param index the position index.
     * @param val the inserted val.
     * @return the result vec internal; it may be {@code this}.
     */
    @Override
    @CheckReturnValue
    public VecInternal insert(int index, Val val) {
        Preconds.checkPosIndex(index, capa());
        int orgSize = this.size;
        if (index > orgSize) {
            // no op when index is out of bounds of the size
            return this;
        }
        int newSize = orgSize + 1;
        if (newSize <= capa()) {
            int moved = orgSize - index;
            slideRight(index, index + 1, moved);
            VALS_VH.setRelease(this.vals, index, val);
            this.size = newSize;
            return this;
        } else {
            int newCapa = calcCapa(newSize);
            Val[] newVals = new Val[newCapa];
            copyOut(0, index, newVals, 0);
            newVals[index] = val;
            copyOut(index, orgSize - index, newVals, index + 1);
            Arrays.fill(newVals, newSize, newVals.length, vm.nada);
            return new MutableVecInternal(vm, newVals, newSize);
        }
    }

    /**
     * Inserts the elements of {@code added} to the position index.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= index &lt;= capa.</p>
     *
     * @param index the position index.
     * @param added the added vec internal.
     * @return the result vec internal; it may be {@code this}.
     */
    @Override
    @CheckReturnValue
    public VecInternal insertAll(int index, VecInternal added) {
        Preconds.checkPosIndex(index, capa());
        int orgSize = this.size;
        if (index > orgSize) {
            // no op when index is out of bounds of the size
            return this;
        }
        int addedSize = added.size();
        int newSize = orgSize + addedSize;
        int moved = orgSize - index;
        if (newSize <= capa()) {
            slideRight(index, index + addedSize, moved);
            added.copyOutRelease(0, addedSize, this.vals, index);
            this.size = newSize;
            return this;
        } else {
            int newCapa = calcCapa(newSize);
            Val[] newVals = new Val[newCapa];
            copyOut(0, index, newVals, 0);
            added.copyOut(0, addedSize, newVals, index);
            copyOut(index, moved, newVals, index + addedSize);
            Arrays.fill(newVals, newSize, newVals.length, vm.nada);
            return new MutableVecInternal(vm, newVals, newSize);
        }
    }

    /**
     * Inserts the elements in the range of {@code src} to the position index.
     *
     * <p>This is a <a href="https://srfi.schemers.org/srfi-1/srfi-1.html#LinearUpdateProcedures">
     * linear update</a> make.</p>
     *
     * <p>Precondition: 0 &lt;= index &lt;= capa, 0 &lt;= from &lt;= to &lt; src.length.</p>
     *
     * @param index the position index.
     * @param src array containing the added vals.
     * @param from the from index of the src.
     * @param to the to index of the src.
     * @return the result vec internal; it may be {@code this}.
     */
    @Override
    @CheckReturnValue
    public VecInternal insertRange(int index, Val[] src, int from, int to) {
        Preconds.checkPosIndex(index, capa());
        Preconds.checkRange(from, to, src.length);
        int orgSize = this.size;
        if (index > orgSize) {
            // no op when index is out of bounds of the size
            return this;
        }
        int addedSize = to - from;
        int newSize = orgSize + addedSize;
        int moved = orgSize - index;
        if (newSize <= capa()) {
            slideRight(index, index + addedSize, moved);
            copyIn(src, from, addedSize, index);
            this.size = newSize;
            return this;
        } else {
            int newCapa = calcCapa(newSize);
            Val[] newVals = new Val[newCapa];
            copyOut(0, index, newVals, 0);
            System.arraycopy(src, from, newVals, index, addedSize);
            copyOut(index, moved, newVals, index + addedSize);
            Arrays.fill(newVals, newSize, newVals.length, vm.nada);
            return new MutableVecInternal(vm, newVals, newSize);
        }
    }

    /**
     * Calculates sufficient capacity greater than or equal to {@code size}.
     */
    private static int calcCapa(int size) {
        return Math.max(size, (int) (size * 1.25));
    }

}

// vim: et sw=4 sts=4 fdm=marker
