package org.kink_lang.kink;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.StringJoiner;

import org.kink_lang.kink.internal.function.ThrowingFunction3;
import org.kink_lang.kink.internal.function.ThrowingFunction4;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.internal.num.NumOperations;

/**
 * The helper for bin vals.
 */
public class BinHelper {

    /** The vm. */
    private final Vm vm;

    /** Shared vars of bins. */
    SharedVars sharedVars;

    /** The maximum valid size of bins. Modifiable by unit tests. */
    int maxSize = Integer.MAX_VALUE;

    /**
     * Constructs the helper.
     */
    BinHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Returns a bin val, copying the bytes.
     *
     * @param bytes the bytes.
     * @return a bin val, copying the bytes.
     */
    public BinVal of(byte[] bytes) {
        return new BinVal(vm, Arrays.copyOf(bytes, bytes.length));
    }

    /**
     * Returns a bin val,
     * copyin the bytes inclusively from {@code from}, exclusively to {@code to}.
     *
     * @param bytes the bytes.
     * @param from the inclusive start index in {@code bytes}.
     * @param to the exclusive end index in {@code bytes}.
     * @return a bin val, copying the bytes.
     */
    public BinVal of(byte[] bytes, int from, int to) {
        byte[] copy = Arrays.copyOfRange(bytes, from, to);
        return new BinVal(vm, copy);
    }

    /**
     * Returns the maximum valid size of bins.
     */
    int getMaxSize() {
        return this.maxSize;
    }

    /**
     * Initializes the helper.
     */
    void init() {
        Map<Integer, Val> vars = new HashMap<>();
        addMethod(vars, "Bin", "empty?", "", 0, (c, desc, bin) -> vm.bool.of(bin.size() == 0));
        addMethod(vars, "Bin", "size", "", 0, (c, desc, bin) -> vm.num.of(bin.size()));
        addMethod(vars, "Bin", "get", "(Ind)", 1, this::getMethod);
        addMethod(vars, "Bin", "slice", "(From_pos To_pos)", 2, this::sliceMethod);
        addTakeDropMethod(vars, "take_front", 0, 0, 1, 0);
        addTakeDropMethod(vars, "take_back", -1, 1, 0, 1);
        addTakeDropMethod(vars, "drop_front", 1, 0, 0, 1);
        addTakeDropMethod(vars, "drop_back", 0, 0, -1, 1);
        addBinaryOp(vars, "Bin", "op_add", "Arg_bin", this::opAddMethod);
        addMethod(vars, "Bin", "op_mul", "(Count)", 1, this::opMulMethod);
        addBinaryOp(vars, "Bin", "op_lt", "Arg_bin",
                (c, desc, x, y) -> vm.bool.of(x.compareTo(y) < 0));
        addBinaryOp(vars, "Bin", "op_eq", "Arg_bin", (c, desc, x, y) -> vm.bool.of(x.equals(y)));
        addMethod(vars, "Bin", "repr", "", 0, this::reprMethod);
        this.sharedVars = vm.sharedVars.of(vars);
    }

    // Bin.get(Index) {{{1

    /**
     * Implementation of Bin.get(Index).
     */
    private HostResult getMethod(CallContext c, String desc, BinVal bin) {
        if (! (c.arg(0) instanceof NumVal indVal)) {
            return c.call(vm.graph.raiseFormat("{}: Ind must be a num, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        int ind = NumOperations.getElemIndex(indVal.bigDecimal(), bin.size());
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat("{}: Ind must be an int [0, {}), but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.str.of(Integer.toString(bin.size()))),
                        vm.graph.repr(indVal)));
        }
        return vm.num.of(Byte.toUnsignedInt(bin.get(ind)));
    }

    // }}}1

    // Bin.slice(From To) {{{1

    /**
     * Implementation of Bin.slice(From_pos To_pos).
     */
    private HostResult sliceMethod(CallContext c, String desc, BinVal bin) {
        if (! (c.arg(0) instanceof NumVal fromVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: From_pos must be an int num, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        BigDecimal fromDec = fromVal.bigDecimal();

        if (! (c.arg(1) instanceof NumVal toVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: To_pos must be an int num, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(1))));
        }
        BigDecimal toDec = toVal.bigDecimal();

        if (! NumOperations.isRangePair(fromDec, toDec, bin.size())) {
            return c.call(vm.graph.raiseFormat(
                        "{}: required range in [0, {}] but got [{}, {}]",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(vm.num.of(bin.size())),
                        vm.graph.repr(fromVal),
                        vm.graph.repr(toVal)));
        }
        int from = fromDec.intValueExact();
        int to = toDec.intValueExact();
        return bin.slice(from, to);
    }

    // }}}1

    // Bin.take, take_back, drop_front, drop_back {{{1

    /**
     * Implementation of Bin.take_front, take_back, drop_front, drop_back.
     */
    private void addTakeDropMethod(
            Map<Integer, Val> vars,
            String methodName,
            int fromNumCoef, int fromSizeCoef, int toNumCoef, int toSizeCoef) {
        addMethod(vars, "Bin", methodName, "(N)", 1, (c, desc, bin) -> {
            if (! (c.arg(0) instanceof NumVal numVal)) {
                return c.call(vm.graph.raiseFormat("{}: N must be an int num, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(c.arg(0))));
            }
            BigDecimal numDec = numVal.bigDecimal();
            int size = bin.size();
            int num = NumOperations.getPosIndex(numDec, size);
            if (num < 0) {
                return c.call(vm.graph.raiseFormat(
                            "{}: N must be an int num in [0, {}], but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.num.of(size)),
                            vm.graph.repr(numVal)));
            }
            int from = fromNumCoef * num + fromSizeCoef * size;
            int to = toNumCoef * num + toSizeCoef * size;
            return bin.slice(from, to);
        });
    }

    // }}}1

    // Bin + Another_bin. {{{1

    /**
     * Implementation of Bin + Another_bin.
     */
    private HostResult opAddMethod(CallContext c, String desc, BinVal x, BinVal y) {
        long resultSize = (long) x.size() + y.size();
        if (resultSize > vm.bin.getMaxSize()) {
            return c.call(vm.graph.raiseFormat(
                        "{}: too long result: size {} + size {} is {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(x.size())),
                        vm.graph.of(vm.num.of(y.size())),
                        vm.graph.of(vm.num.of(resultSize))));
        }
        return x.concat(y);
    }

    // }}}1

    // Bin * Count {{{1

    /**
     * Implementation of Bin * Count.
     */
    private HostResult opMulMethod(CallContext c, String desc, BinVal bin) {
        BigInteger count = NumOperations.getExactBigInteger(c.arg(0));
        if (count == null || count.signum() < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Count must be int num >=0, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }

        if (bin.size() == 0) {
            return bin;
        }

        BigInteger resultSize = count.multiply(BigInteger.valueOf(bin.size()));
        if (resultSize.compareTo(BigInteger.valueOf(vm.bin.getMaxSize())) > 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: too long result: size {} * Count {} is {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(bin.size())),
                        vm.graph.repr(c.arg(0)),
                        vm.graph.of(vm.num.of(resultSize))));
        }

        int countInt = count.intValueExact();
        byte[] bytes = bin.bytes();
        byte[] result = new byte[countInt * bin.size()];
        for (int i = 0; i < countInt; ++ i) {
            System.arraycopy(bytes, 0, result, i * bytes.length, bytes.length);
        }
        return vm.bin.of(result);
    }

    // }}}1

    // Bin.repr {{{1

    /**
     * Implementation of Bin.repr.
     */
    private HostResult reprMethod(CallContext c, String desc, BinVal bin) {
        StringJoiner sj = new StringJoiner(" ", "(", ")");
        sj.add("bin");
        for (byte b : bin.bytes()) {
            String s = Integer.toString(Byte.toUnsignedInt(b), 16);
            sj.add((s.length() == 1 ? "0x0" : "0x") + s);
        }
        return vm.str.of(sj.toString());
    }

    // }}}1

    /**
     * Adds a method to vars.
     */
    private void addMethod(
            Map<Integer, Val> vars,
            String recvDesc,
            String methodName,
            String argsDesc,
            int arity,
            ThrowingFunction3<CallContext, String, BinVal, HostResult> action) {
        String desc = String.format(Locale.ROOT, "%s.%s%s", recvDesc, methodName, argsDesc);
        int handle = vm.sym.handleFor(methodName);
        var fun = vm.fun.make(desc).take(arity).action(c -> {
            if (! (c.recv() instanceof BinVal recv)) {
                return c.call(vm.graph.raiseFormat("{}: {} must be bin, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(recvDesc)),
                            vm.graph.repr(c.recv())));
            }
            return action.apply(c, desc, recv);
        });
        vars.put(handle, fun);
    }

    /**
     * Adds a binary operator to vars.
     */
    private void addBinaryOp(
            Map<Integer, Val> vars,
            String recvDesc,
            String methodName,
            String argDesc,
            ThrowingFunction4<CallContext, String, BinVal, BinVal, HostResult> action) {
        String argsDesc = "(" + argDesc + ")";
        addMethod(vars, recvDesc, methodName, argsDesc, 1, (c, desc, recv) -> {
            if (! (c.arg(0) instanceof BinVal arg)) {
                return c.call(vm.graph.raiseFormat("{}: {} must be bin, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(argDesc)),
                            vm.graph.repr(c.arg(0))));
            }
            return action.apply(c, desc, recv, arg);
        });
    }

}

// vim: et sw=4 sts=4 fdm=marker
