package org.kink_lang.kink;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.StringJoiner;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * A bin val, which represents an immutable array of bytes.
 *
 * <p>A bin val holds Java byte arrays as a binary tree in order to avoid
 * allocation of the entire array on concatenation.
 * The byte array is substantiated on demand.</p>
 */
public class BinVal extends Val implements Comparable<BinVal> {

    /**
     * A node containing the bytes.
     * It may be replaced to an equivalent node to simplify the representation.
     *
     * Access to this field is not synchronized.
     * It is safe because:
     *
     * 1. all the implementation of BytesNode are immutable, and
     * 2. reading a stale value does not cause inconsistent result,
     *    although it may cause otherwise unneeded reinvocation of {@link BytesTree#asLeaf()}.
     */
    private BytesNode bytesNode;

    /**
     * Constructs a bin val, not copying the bytes.
     *
     * @param vm the vm.
     * @param bytes the source bytes.
     */
    BinVal(Vm vm, byte[] bytes) {
        this(vm, new BytesLeaf(bytes));
    }

    /**
     * Constructs a bin val.
     */
    private BinVal(Vm vm, BytesNode bytesNode) {
        super(vm, null);
        Preconds.checkArg(bytesNode.size() <= vm.bin.getMaxSize(),
                "size must not exceed getMaxSize()");
        this.bytesNode = bytesNode;
    }

    /**
     * Returns the size of the bin.
     *
     * @return the size of the bin.
     */
    public int size() {
        return this.bytesNode.size();
    }

    /**
     * Returns the copied bytes of the bin.
     *
     * @return the copied bytes of the bin.
     */
    public byte[] bytes() {
        return Arrays.copyOf(rawBytes(), size());
    }

    /**
     * Copy the range of this bin into {@code bytes}
     * at the {@code at} index.
     *
     * @param from the inclusive start index.
     * @param to the exclusive end index.
     * @param bytes the destination bytes.
     * @param at the start index of the destination bytes.
     */
    public void copyToBytes(int from, int to, byte[] bytes, int at) {
        int length = to - from;
        System.arraycopy(rawBytes(), from, bytes, at, length);
    }

    /**
     * Makes a real byte array if not made,
     * and returns it.
     */
    private byte[] rawBytes() {
        BytesLeaf leaf = this.bytesNode.asLeaf();
        this.bytesNode = leaf;
        return leaf.getBytes();
    }

    /**
     * Returns the byte at the ind.
     *
     * @param ind the index of the byte.
     * @return the byte at the ind.
     * @throws IndexOutOfBoundsException if {@code ind} is out of the range of the bin.
     */
    public byte get(int ind) {
        Preconds.checkElemIndex(ind, size());
        return rawBytes()[ind];
    }

    /**
     * Returns a read-only {@link ByteBuffer} of the content of this bin.
     *
     * <p>The capacity of the result ByteBuffer is the size of the bin.
     * The limit of the result ByteBuffer is set to the size of the bin.
     * The position of the result ByteBuffer is set to 0.</p>
     *
     * @return a read-only {@link ByteBuffer} of the content of this bin.
     */
    public ByteBuffer readOnlyByteBuffer() {
        ByteBuffer mutBuf = ByteBuffer.wrap(rawBytes());
        return mutBuf.asReadOnlyBuffer();
    }

    /**
     * Returns a sliced bin between {@code from} (inclusive) and {@code to} (exclusive).
     *
     * @param from the inclusive index where the result is sliced from.
     * @param to the exclusive index where the result is sliced to.
     * @return a sliced bin.
     * @throws IndexOutOfBoundsException
     *      if {@code from} or {@code to} are out of the range of the bin.
     * @throws IllegalArgumentException if {@code from} is larger than {@code to}.
     */
    public BinVal slice(int from, int to) {
        Preconds.checkRange(from, to, size());
        return new BinVal(vm, Arrays.copyOfRange(rawBytes(), from, to));
    }

    /**
     * Returns a negative number if {@code this} is less than {@code arg},
     * 0 if {@code this} is equivalent to {@code arg},
     * or a positive number if {@code this} is greater than {@code arg}.
     *
     * <p>Comparison is done for the bytes of {@code this} and {@code arg},
     * treating the bytes as unsigned numbers.</p>
     *
     * <p>Comparison is agnostic with {@code vm} field.</p>
     *
     * @param arg the arg bin to compare with.
     * @return the comparison number.
     */
    @Override
    public int compareTo(BinVal arg) {
        return Arrays.compareUnsigned(rawBytes(), arg.rawBytes());
    }

    /**
     * Returns a bin val which concatenates {@code this} and {@code tail}.
     *
     * @param tail the bin which is appended to the result.
     * @return a bin val which concatenates {@code this} and {@code tail}.
     */
    public BinVal concat(BinVal tail) {
        Preconds.checkArg((long) size() + tail.size() <= vm.bin.getMaxSize(),
                "result size exceeds the max");

        if (size() == 0) {
            return tail;
        } else if (tail.size() == 0) {
            return this;
        }

        BytesNode left = this.bytesNode;
        BytesNode right = tail.bytesNode;
        int leafCountSum = left.getLeafCount() + right.getLeafCount();

        // limit the count of leaves to limit capacity of the work stack in BytesLeaf#concat
        return leafCountSum < 128
            ? new BinVal(vm, new BytesTree(left, right))
            : new BinVal(vm, BytesLeaf.concat(List.of(right, left)));

    }

    /**
     * Returns true if and only if {@code arg} is a bin val
     * and has the same vm and the equivalent bytes with {@code this}.
     *
     * @param arg the arg object to compare with.
     * @return true if and only if {@code arg} is a bin val
     *      and has the same vm and the equivalent bytes with {@code this}.
     */
    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof BinVal
            && this.vm.equals(((BinVal) arg).vm)
            && Arrays.equals(rawBytes(), ((BinVal) arg).rawBytes());
    }

    @Override
    public int hashCode() {
        return this.vm.hashCode() * 101
            + Arrays.hashCode(this.rawBytes());
    }

    @Override
    public String toString() {
        StringJoiner sj = new StringJoiner(" ", "BinVal(", ")");
        for (byte b : rawBytes()) {
            String s = Integer.toString(Byte.toUnsignedInt(b), 16);
            sj.add((s.length() == 1 ? "0x0" : "0x") + s);
        }
        return sj.toString();
    }

    @Override
    SharedVars sharedVars() {
        return vm.bin.sharedVars;
    }

    /**
     * A node of a binary tree holding the bytes.
     */
    private static abstract class BytesNode {

        /**
         * Returns the size of the bytes.
         */
        abstract int size();

        /**
         * Returns the count of the leaves.
         */
        abstract int getLeafCount();

        /**
         * Returns the leaf equivalent to the tree.
         */
        abstract BytesLeaf asLeaf();

    }

    /**
     * Node with two branches.
     */
    private static class BytesTree extends BytesNode {

        /** The size of the bytes. */
        private final int size;

        /** The count of the leaves under the branch. */
        private final int leafCount;

        /** The left branch. */
        private final BytesNode left;

        /** The right branch. */
        private final BytesNode right;

        /**
         * Constructs a node with two branches.
         */
        BytesTree(BytesNode left, BytesNode right) {
            this.size = left.size() + right.size();
            this.leafCount = left.getLeafCount() + right.getLeafCount();
            this.left = left;
            this.right = right;
        }

        @Override
        int size() {
            return this.size;
        }

        @Override
        int getLeafCount() {
            return this.leafCount;
        }

        @Override
        BytesLeaf asLeaf() {
            return BytesLeaf.concat(List.of(this));
        }

    }

    /**
     * A leaf node.
     */
    private static class BytesLeaf extends BytesNode {

        /** The bytes. */
        private final byte[] bytes;

        /**
         * Constructs a leaf.
         */
        BytesLeaf(byte[] bytes) {
            this.bytes = bytes;
        }

        @Override
        int size() {
            return this.bytes.length;
        }

        @Override
        int getLeafCount() {
            return 1;
        }

        @Override
        BytesLeaf asLeaf() {
            return this;
        }

        /**
         * Returns the bytes.
         */
        byte[] getBytes() {
            return this.bytes;
        }

        /**
         * Concatenates the nodes to a leaf.
         *
         * @param initialStack leftmost on the top, rightmost on the bottom.
         */
        static BytesLeaf concat(List<BytesNode> initialStack) {
            int size = initialStack.stream().mapToInt(BytesNode::size).sum();
            List<BytesNode> stack = new ArrayList<>(initialStack);
            int ind = 0;
            byte[] result = new byte[size];
            while (! stack.isEmpty()) {
                BytesNode node = stack.remove(stack.size() - 1);
                if (node instanceof BytesLeaf) {
                    BytesLeaf leaf = (BytesLeaf) node;
                    System.arraycopy(leaf.getBytes(), 0, result, ind, leaf.size());
                    ind += leaf.size();
                } else {
                    BytesTree tree = (BytesTree) node;
                    stack.add(tree.right);
                    stack.add(tree.left);
                }
            }
            return new BytesLeaf(result);
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
