package org.kink_lang.kink;

import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

import java.util.Map;

/**
 * The helper for bool vals.
 */
public class BoolHelper {

    /** The vm. */
    private final Vm vm;

    /** Kink true. */
    public final Val trueVal;

    /** Kink false. */
    public final Val falseVal;

    /** Shared vars for bool vals. */
    private SharedVars boolVars;

    /**
     * Constructs the helper.
     */
    BoolHelper(Vm vm) {
        this.vm = vm;
        this.trueVal = new Val(vm, null) {
            @Override public String toString() { return "trueVal"; }
            @Override
            SharedVars sharedVars() {
                return vm.bool.boolVars;
            }
        };
        this.falseVal = new Val(vm, null) {
            @Override public String toString() { return "falseVal"; }
            @Override
            SharedVars sharedVars() {
                return vm.bool.boolVars;
            }
        };
    }

    /**
     * Returns a bool val from the Java boolean.
     *
     * @param bl the Java boolean.
     * @return a bool val.
     */
    public Val of(boolean bl) {
        return bl ? trueVal : falseVal;
    }

    /**
     * Returns true if the val is a bool val.
     *
     * @param val the val.
     * @return true if the val is a bool val.
     */
    public boolean isBool(Val val) {
        return val == this.trueVal || val == this.falseVal;
    }

    /**
     * Initialize bool vals.
     */
    void init() {
        this.boolVars = vm.sharedVars.of(Map.of(
                    vm.sym.handleFor("op_eq"),
                    vm.fun.make("Bool1.op_eq(Bool2)").take(1).action(this::opEqMethod),

                    vm.sym.handleFor("repr"),
                    vm.fun.make("Bool.repr").take(0).action(this::reprMethod)
                    ));
    }

    /**
     * Implementation of Bool.op_eq.
     */
    private HostResult opEqMethod(CallContext c) {
        Val arg = c.arg(0);
        if (! vm.bool.isBool(arg)) {
            return c.call(vm.graph.raiseFormat(
                        "Bool1.op_eq(Bool2): Bool2 must be a bool, but got {}",
                        vm.graph.repr(arg)));
        }
        return vm.bool.of(arg.equals(c.recv()));
    }

    /**
     * Implementation of Bool.repr.
     */
    private HostResult reprMethod(CallContext c) {
        Val recv = c.recv();
        if (! vm.bool.isBool(recv)) {
            return c.call(vm.graph.raiseFormat(
                        "Bool.repr: Bool must be a bool, but got {}",
                        vm.graph.repr(recv)));
        }
        return vm.str.of(recv == vm.bool.trueVal ? "true" : "false");
    }

}

// vim: et sw=4 sts=4 fdm=marker
