package org.kink_lang.kink;

import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.util.List;
import java.util.Locale;

import org.kink_lang.kink.internal.control.Control;

/**
 * A fun implementation which delegates {@code run} method to another fun.
 *
 * This implementation is used with letrec.
 */
class DelegatingFunVal extends FunVal {

    /** Wrapped fun to which {@code run} method is delegated; accessed via WRAPPED_VH. */
    @SuppressWarnings("PMD.UnusedPrivateField") // PMD does not see VarHandle
    private FunVal wrapped;

    /** The var handle for the wrapped field. */
    private static final VarHandle WRAPPED_VH = Control.runWrappingThrowable(
            () -> MethodHandles.lookup()
            .findVarHandle(DelegatingFunVal.class, "wrapped", FunVal.class));

    /**
     * Constructs a fun with no wrapped fun..
     */
    DelegatingFunVal(Vm vm) {
        super(vm);
    }

    /**
     * Emplaces the wrapped fun.
     */
    void wrap(FunVal wrapped) {
        WRAPPED_VH.setRelease(this, wrapped);
    }

    /**
     * Returns the wrapped fun.
     *
     * This method must be invoked after {@code wrap} method is invoked.
     */
    FunVal getWrapped() {
        return (FunVal) WRAPPED_VH.getAcquire(this);
    }

    @Override
    String getRepr() {
        var wrapped = getWrapped();
        return String.format(Locale.ROOT, "(fun variant=%s wrapped=%s val_id=%s)",
                getClass().getSimpleName(),
                wrapped == null ? "null" : wrapped.getRepr(),
                identity());
    }

    @Override
    void run(StackMachine stackMachine) {
        getWrapped().run(stackMachine);
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "DelegatingFunVal(%s)", getWrapped());
    }

    /**
     * Returns th eproperties of this fun.
     */
    private List<Object> getProperties() {
        return List.of(this.vm, getWrapped());
    }

    @Override
    public int hashCode() {
        return getProperties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof DelegatingFunVal
            && getProperties().equals(((DelegatingFunVal) arg).getProperties());
    }

}

// vim: et sw=4 sts=4 fdm=marker
