package org.kink_lang.kink;

import java.math.BigDecimal;
import java.util.List;
import java.util.Locale;

/**
 * A num val.
 *
 * <p>A num val consists of two integers: mantissa and scale.
 * The num val represents an decimal number (mantissa * 10**(-scale)).
 * Kink runtimes must support arbitrary range for mantissa as far as the memory is not exhausted,
 * and at least the range [-32767, +32767] for scale.</p>
 *
 * <p>In this implementation, a num val is backed by a Java BigDecimal.
 * The mantissa is the “unscaled value” of BigDecimal,
 * and the scale is the “scale” of BigDecimal.</p>
 */
public class NumVal extends Val {

    /** The big decimal number. */
    private final BigDecimal bigDecimal;

    /**
     * Constructs a num val from a BigDecimal with 0 or positive scale.
     *
     * @param vm the vm.
     * @param bigDecimal the big decimal number.
     * @throws IllegalArgumentException when the scale is negative.
     */
    NumVal(Vm vm, BigDecimal bigDecimal) {
        super(vm, null);
        this.bigDecimal = bigDecimal;
    }

    /**
     * Returns the big decimal number.
     *
     * @return the big decimal number.
     */
    public BigDecimal bigDecimal() {
        return this.bigDecimal;
    }

    @Override
    SharedVars sharedVars() {
        return vm.num.sharedVars;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "NumVal(%s)", this.bigDecimal);
    }

    /**
     * Returns properties which determine equality of the type.
     */
    private List<Object> properties() {
        return List.of(this.vm, this.bigDecimal);
    }

    @Override
    public int hashCode() {
        return properties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof NumVal argNum
            && properties().equals(argNum.properties());
    }

    /**
     * Returns whether {@code val} is a num with no own var.
     *
     * @return whether {@code val} is a num with no own var.
     */
    static boolean isPlainNum(Val val) {
        return val instanceof NumVal num
            && num.noOwnVar();
    }

}

// vim: et sw=4 sts=4 fdm=marker
