package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.stream.IntStream;

import org.objectweb.asm.Label;
import org.objectweb.asm.commons.GeneratorAdapter;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * State of generation of bytecode.
 */
public class BytecodeGenState {

    /** bytecode generating method visitor. */
    private final GeneratorAdapter ga;

    /** The list of insns. */
    private final List<Insn> insns;

    /** Mapping from string key to local var id. */
    private Map<String, Integer> locals;

    /** Mapping from label key to label. */
    private Map<String, Label> labels;

    /** Mapping of switch-case-default labels. */
    private final Map<SwitchLabelKey, Label> switchLabels;

    /**
     * Constructs state with the generator method visitor.
     *
     * @param ga Generator method visitor
     * @param insns the list of insns.
     */
    public BytecodeGenState(GeneratorAdapter ga, List<Insn> insns) {
        this.ga = ga;
        this.insns = List.copyOf(insns);
        this.locals = new HashMap<>();
        this.labels = new HashMap<>();
        this.switchLabels = new HashMap<>();
    }

    /**
     * Sets the mapping from string name to local var id.
     */
    void setLocals(Map<String, Integer> locals) {
        this.locals = locals;
    }

    /**
     * Sets the mapping from label name to label.
     */
    void setLabels(Map<String, Label> labels) {
        this.labels = labels;
    }

    /**
     * Returns the generator adapter.
     *
     * @return the generator adapter.
     */
    public GeneratorAdapter ga() {
        return this.ga;
    }

    /**
     * Returns the label corresponding to {@code labelKey}.
     *
     * If {@code labelKey} is not yet registered, registers it.
     *
     * @param labelKey the name of the label.
     * @return the label corresponding to {@code labelKey}.
     */
    Label labelFor(String labelKey) {
        if (! this.labels.containsKey(labelKey)) {
            this.labels.put(labelKey, this.ga.newLabel());
        }
        return this.labels.get(labelKey);
    }

    /**
     * Associates the local var index with {@code localKey}.
     *
     * @param localKey the name of the local var.
     * @param local the local var index.
     */
    void registerLocal(String localKey, int local) {
        Preconds.checkArg(! this.locals.containsKey(localKey), "conflicting local: " + localKey);
        this.locals.put(localKey, local);
    }

    /**
     * Returns the local var index of {@code localKey}.
     *
     * @param localKey the name of the local var.
     * @return the local var index.
     */
    int getLocal(String localKey) {
        Preconds.checkArg(this.locals.containsKey(localKey), "not found: " + localKey);
        return this.locals.get(localKey);
    }

    /**
     * Generates tableswitch insn.
     *
     * <p>The method assumes case numes are not sparse,
     * because the implementation can be much simpler with the assumption,
     * and sparse case nums don't appear in Kink bytecode generation.</p>
     */
    void generateSwitch(String switchKey) {
        var minMax = caseMinMax(switchKey);
        int min = minMax.min();
        int max = minMax.max();

        Label[] caseLabels = IntStream.rangeClosed(min, max)
            .mapToObj(n -> new Label())
            .toArray(Label[]::new);
        IntStream.rangeClosed(min, max)
            .mapToObj(n -> new SwitchLabelKey.Case(switchKey, n))
            .forEach(key -> this.switchLabels.put(key, caseLabels[key.num() - min]));

        Label defLabel = ga().newLabel();
        this.switchLabels.put(new SwitchLabelKey.Default(switchKey), defLabel);

        ga().visitTableSwitchInsn(min, max, defLabel, caseLabels);
    }

    /**
     * Min and max case nums.
     *
     * @param min the min case num.
     * @param max the max case num.
     */
    private record MinMax(int min, int max) {};

    /**
     * Returns the min and max case nums for the switch key.
     */
    private MinMax caseMinMax(String switchKey) {
        int[] nums = this.insns.stream()
            .filter(Insn.Case.class::isInstance)
            .map(Insn.Case.class::cast)
            .filter(c -> c.switchKey().equals(switchKey))
            .mapToInt(c -> c.num())
            .sorted()
            .toArray();
        if (nums.length == 0) {
            throw new IllegalStateException(
                    String.format(Locale.ROOT, "no case for switch key %s", switchKey));
        }
        for (int prev = 0, next = 1; next < nums.length; ++ prev, ++ next) {
            if (nums[prev] + 1 != nums[next]) {
                throw new IllegalStateException(
                        "case keys must not be sparse, but was: " + switchKey);
            }
        }
        return new MinMax(nums[0], nums[nums.length - 1]);
    }

    /**
     * Returns the case label for the switch.
     */
    Label caseLabel(String switchKey, int num) {
        return this.switchLabels.get(new SwitchLabelKey.Case(switchKey, num));
    }

    /**
     * Returns the default label for the switch.
     */
    Label defaultLabel(String switchKey) {
        return this.switchLabels.get(new SwitchLabelKey.Default(switchKey));
    }

    /**
     * Key for switchLabels.
     */
    sealed interface SwitchLabelKey {

        /**
         * Key for case label.
         *
         * @param switchKey the key of the switch.
         * @param num the num for the case.
         */
        record Case(String switchKey, int num) implements SwitchLabelKey {}

        /**
         * Key for default label.
         *
         * @param switchKey the key of the switch.
         */
        record Default(String switchKey) implements SwitchLabelKey {}
    }

}

// vim: et sw=4 sts=4 fdm=marker
