package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import org.kink_lang.kink.internal.program.itree.FastFunItree;
import org.objectweb.asm.Type;

import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.program.itree.LocalVar;

/**
 * Generates insns of making val-capturing fast fun.
 */
public class MakeValCaptureFastFunGenerator implements MakeFastFunGenerator {

    /** Generator of lvars access insns of the enclosing context. */
    private final LvarAccessGenerator lvarAccGen;

    /** Compiles funs. */
    private final Function<FastFunItree, JavaClassIr> compile;

    /** Analyzes the storage of funs. */
    private final Function<FastFunItree, AllocationSet> analyzeAllocation;

    /** The accumulator of child JavaClassIr instances. */
    private final ChildJcirAccumulator jcirAccum;

    /**
     * Constructs a generator.
     *
     * @param lvarAccGen generator of lvars access insns of the enclosing context.
     * @param compile compiles funs.
     * @param analyzeAllocation analyzes the storage of funs.
     * @param jcirAccum the accumulator of child JavaClassIr instances.
     */
    public MakeValCaptureFastFunGenerator(
            LvarAccessGenerator lvarAccGen,
            Function<FastFunItree, JavaClassIr> compile,
            Function<FastFunItree, AllocationSet> analyzeAllocation,
            ChildJcirAccumulator jcirAccum) {
        this.lvarAccGen = lvarAccGen;
        this.compile = compile;
        this.analyzeAllocation = analyzeAllocation;
        this.jcirAccum = jcirAccum;
    }

    @Override
    public List<Insn> makeFun(FastFunItree fun) {
        List<LocalVar> free = analyzeAllocation.apply(fun).field();
        int jcirInd = jcirAccum.add(() -> compile.apply(fun));
        return free.isEmpty()
            ? MakeFastFunGenerator.generateCombinator(jcirInd)
            : generateNonCombinator(jcirInd, free);
    }

    /**
     * Generates insns for non-combinator fun.
     */
    private List<Insn> generateNonCombinator(
            int jcirInd,
            List<LocalVar> free) {
        List<Insn> insns = new ArrayList<>();

        // push vm
        insns.add(new Insn.LoadThis());
        insns.add(new Insn.GetField(JavaClassIr.TYPE_BASE, "vm", Type.getType(Vm.class)));

        // push args
        for (LocalVar lv : free) {
            insns.addAll(lvarAccGen.loadLvarAllowNull(lv));
            insns.add(InsnsGenerator.LOAD_CONTPARAM);
        }

        // make fun
        insns.add(InsnsGenerator.invokeMakeFun(free.size(), jcirInd));
        return insns;
    }

}

// vim: et sw=4 sts=4 fdm=marker
