package org.kink_lang.kink.internal.mod.random;

import java.util.HashMap;
import java.util.Map;

import org.kink_lang.kink.SharedVars;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * xoshiro256** PRNG val.
 */
public class PrngVal extends RngVal<Xoshiro256StarStar> {

    /**
     * Constructs a PRNG val.
     */
    PrngVal(Vm vm, SharedVars sharedVars, Xoshiro256StarStar random) {
        super(vm, sharedVars, random, "prng");
    }

    /**
     * Makes a new PRNG val.
     *
     * @param vm the vm.
     * @param random the xoshiro256** rng.
     * @return a new PRNG val.
     */
    public static PrngVal of(Vm vm, Xoshiro256StarStar random) {
        PrngHelper helper = PrngHelper.of(vm);
        return new PrngVal(vm, helper.sharedVars, random);
    }

    /**
     * Helper of thrower vals.
     */
    static class PrngHelper {

        /** Vm. */
        private final Vm vm;

        /** Shared vars of thrower vals. */
        final SharedVars sharedVars;

        /**
         * Constructs a helper for the vm.
         */
        private PrngHelper(Vm vm) {
            this.vm = vm;
            this.sharedVars = makeSharedVars();
        }

        /**
         * Returns the helper instance of the vm.
         */
        static PrngHelper of(Vm vm) {
            return vm.component.getOrRegister(PrngHelper.class, PrngHelper::new);
        }

        /**
         * Makes the set of shared vars for the helper.
         */
        private SharedVars makeSharedVars() {
            RngVal.RngHelper rngHelper = RngVal.RngHelper.of(vm);
            Map<Integer, Val> map = new HashMap<>(rngHelper.varMap);
            map.put(vm.sym.handleFor("state"),
                    vm.fun.make("Prng.state").take(0).action(this::stateMeethod));
            return vm.sharedVars.of(map);
        }

        /**
         * Implementation of Prng.state.
         */
        private HostResult stateMeethod(CallContext c) {
            Val recv = c.recv();
            if (! (recv instanceof PrngVal)) {
                return c.call(vm.graph.raiseFormat(
                            "Prng.state: Prng must be a prng val, but got {}",
                            vm.graph.repr(recv)));
            }
            PrngVal prng = (PrngVal) recv;
            Xoshiro256StarStar random = prng.getRandom();
            long[] stateLongs = random.getState();
            byte[] state = new byte[Long.BYTES * stateLongs.length];
            for (int i = 0; i < stateLongs.length; ++ i) {
                long num = stateLongs[i];
                for (int j = 0; j < Long.BYTES; ++ j) {
                    state[i * Long.BYTES + j] = (byte) (num >>> (Long.BYTES - j - 1) * Byte.SIZE);
                }
            }
            return vm.bin.of(state);
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
