package org.kink_lang.kink.internal.num;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.OptionalInt;
import java.util.OptionalLong;

import org.kink_lang.kink.NumVal;
import org.kink_lang.kink.Val;

/**
 * Number operations used in Kink.
 */
public final class NumOperations {

    /**
     * Cannot be instantiated.
     */
    NumOperations() {
        throw new UnsupportedOperationException("cannot be instantiated");
    }

    /**
     * Returns num.hash.
     *
     * @param num the num to hash.
     * @return the hash used in Kink.
     */
    public static int hash(BigDecimal num) {
        num = num.stripTrailingZeros();
        if (num.scale() < 0) {
            num = num.setScale(0);
        }
        return num.hashCode();
    }

    /**
     * Returns the exact int value of the num,
     * if the num is a proper element index in the sequence which has the size {@code size},
     * otherwise returns -1.
     *
     * @param num the num.
     * @param size the size of a sequence.
     * @return the exact int value of the num, or -1.
     */
    public static int getElemIndex(BigDecimal num, int size) {
        if (num.scale() != 0) {
            return -1;
        }

        try {
            int intVal = num.intValueExact();
            return 0 <= intVal && intVal < size ? intVal : -1;
        } catch (ArithmeticException ex) {
            return -1;
        }
    }

    /**
     * Returns the exact int value of the val,
     * if the val is a NumVal and it represents a valid element index exactly
     * in the sequence which has the size {@code size},
     * otherwise returns -1.
     *
     * @param val the val from which the index is extracted.
     * @param size the size of a sequence.
     * @return the exact int number of the val, or -1.
     */
    public static int getElemIndex(Val val, int size) {
        if (! (val instanceof NumVal)) {
            return -1;
        }
        BigDecimal dec = ((NumVal) val).bigDecimal();
        return getElemIndex(dec, size);
    }

    /**
     * Returns the exact int value of the num,
     * if the num is a proper pos index in the sequence which has the size {@code size},
     * otherwise returns -1.
     *
     * @param num the num.
     * @param size the size of a sequence.
     * @return the exact int value of the num, or -1.
     */
    public static int getPosIndex(BigDecimal num, int size) {
        if (num.scale() != 0) {
            return -1;
        }

        try {
            int intVal = num.intValueExact();
            return 0 <= intVal && intVal <= size ? intVal : -1;
        } catch (ArithmeticException ex) {
            return -1;
        }
    }

    /**
     * Returns the exact int value of the val,
     * if the val is a NumVal and it represents a valid position index exactly
     * in the sequence which has the size {@code size},
     * otherwise returns -1.
     *
     * @param val the val from which the index is extracted.
     * @param size the size of a sequence.
     * @return the exact int number of the val, or -1.
     */
    public static int getPosIndex(Val val, int size) {
        if (! (val instanceof NumVal)) {
            return -1;
        }

        BigDecimal dec = ((NumVal) val).bigDecimal();
        return getPosIndex(dec, size);
    }

    /**
     * Returns an exact int number from the val,
     * if the val is a NumVal and it represents the exact int number.
     *
     * @param val the val from which the int number is extracted.
     * @return the exact int number, or empty.
     */
    public static OptionalInt getExactInt(Val val) {
        if (! (val instanceof NumVal)) {
            return OptionalInt.empty();
        }

        BigDecimal dec = ((NumVal) val).bigDecimal();
        if (dec.scale() != 0) {
            return OptionalInt.empty();
        }

        try {
            return OptionalInt.of(dec.intValueExact());
        } catch (ArithmeticException ae) {
            return OptionalInt.empty();
        }
    }

    /**
     * Returns an exact long number from the val,
     * if the val is a NumVal and it represents the exact long number
     * between {@code min} and {@code max}.
     *
     * @param val the val the long number from which is extracted.
     * @param min the minimum of the range.
     * @param max the maximum of the range.
     * @return the exact long number, or empty.
     */
    public static OptionalLong getExactLongBetween(Val val, long min, long max) {
        if (! (val instanceof NumVal)) {
            return OptionalLong.empty();
        }

        BigDecimal dec = ((NumVal) val).bigDecimal();
        if (dec.scale() != 0) {
            return OptionalLong.empty();
        }

        try {
            long num = dec.longValueExact();
            return min <= num && num <= max ? OptionalLong.of(num) : OptionalLong.empty();
        } catch (ArithmeticException ae) {
            return OptionalLong.empty();
        }
    }

    /**
     * Returns an exact BigInteger from the val,
     * if the val is a NumVal and it represents an int number.
     *
     * @param val the val from which the BigInteger is extracted.
     * @return the exact BigInteger, or null.
     */
    public static BigInteger getExactBigInteger(Val val) {
        if (! (val instanceof NumVal)) {
            return null;
        }

        BigDecimal dec = ((NumVal) val).bigDecimal();
        if (dec.scale() != 0) {
            return null;
        }

        return dec.toBigIntegerExact();
    }

    /**
     * Returns whether {@code from} and {@code to} are proper range indexes of the sequence
     * which has the size {@code size}.
     *
     * @param from the from index.
     * @param to the to index.
     * @param size the size of a sequence.
     * @return true if {@code from} and {@code to} are proper range indexes.
     */
    public static boolean isRangePair(BigDecimal from, BigDecimal to, int size) {
        if (from.scale() != 0 || to.scale() != 0) {
            return false;
        }

        try {
            int fromExact = from.intValueExact();
            int toExact = to.intValueExact();
            return 0 <= fromExact && fromExact <= toExact && toExact <= size;
        } catch (ArithmeticException ex) {
            return false;
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
