package org.kink_lang.kink.internal.program.itree;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Assignment in the form {@code [:A :B :C.opt :D.opt] <- xxx}.
 *
 * @param mandatory the local vars of mandatory params.
 * @param opt       the local vars of optional params.
 * @param rhs       the right-hand-side.
 * @param pos       the pos index of the assignment.
 */
public record OptVecAssignmentItree(List<LocalVar> mandatory,
                                    List<LocalVar> opt,
                                    Itree rhs,
                                    int pos) implements ItreeWithRhs {

    /**
     * Constructs an assignment itree.
     *
     * @param mandatory the local vars of mandatory params.
     * @param opt       the local vars of optional params.
     * @param rhs       the right-hand-side.
     * @param pos       the pos index of the assignment.
     */
    public OptVecAssignmentItree {
        mandatory = List.copyOf(mandatory);
        opt = List.copyOf(opt);
    }

    /**
     * Returns the string representation of the left hand side.
     *
     * @return the string representation of the left hand side.
     */
    public String lhsRepr() {
        return Stream.concat(
                mandatory().stream().map(LocalVar::lhsRepr),
                opt().stream().map(LocalVar::lhsRepr).map(s -> s + ".opt"))
            .collect(Collectors.joining(" ", "[", "]"));
    }

    /**
     * Returns the minimum arity of the left hand side.
     *
     * @return the minimum arity of the left hand side.
     */
    public int minArity() {
        return mandatory().size();
    }

    /**
     * Returns the maximum arity of the left hand side.
     *
     * @return the maximum arity of the left hand side.
     */
    public int maxArity() {
        return minArity() + optCount();
    }

    /**
     * Returns the count of opt parameters.
     *
     * @return the count of opt parameters.
     */
    public int optCount() {
        return opt().size();
    }

    @Override
    public <T> T accept(ItreeVisitor<T> visitor) {
        return visitor.visit(this);
    }

}

// vim: et sw=4 sts=4 fdm=marker
