package org.kink_lang.kink.internal.program.itree;

import java.util.List;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * An itree of sequential evaluation;
 * that corresponds to a not-empty seq node.
 *
 * @param steps the steps of the sequential evaluation; must not be empty.
 * @param pos   the pos of the itree on the program text.
 */
public record SeqItree(List<Itree> steps,
                       int pos) implements Itree {

    /**
     * Constructs an itree of sequential evaluation;
     * that corresponds to a not-empty seq node.
     *
     * <p>The argument steps must not be empty.</p>
     *
     * @param steps the steps of the sequential evaluation; must not be empty.
     * @param pos   the pos of the itree on the program text.
     */
    public SeqItree {
        Preconds.checkArg(!steps.isEmpty(), "steps must not be empty");
        steps = List.copyOf(steps);
    }

    /**
     * Returns the steps excepting the last one.
     *
     * @return the steps excepting the last one.
     */
    public List<Itree> getLeadingSteps() {
        return steps.subList(0, steps.size() - 1);
    }

    /**
     * Returns the last step.
     *
     * @return the last step.
     */
    public Itree getLastStep() {
        return steps.get(steps.size() - 1);
    }

    @Override
    public <T> T accept(ItreeVisitor<T> visitor) {
        return visitor.visit(this);
    }

}

// vim: et sw=4 sts=4 fdm=marker
