package org.kink_lang.kink.internal.str;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;

/**
 * Converter from texts to a sequence of bytes.
 */
public class Encoder {

    /** The backing Charset encoder. */
    private final CharsetEncoder csEncoder;

    /** The char buffer. */
    private final CharBuffer charBuffer;

    /** The byte buffer. */
    private final ByteBuffer byteBuffer;

    /** true after terminate() is called. */
    private boolean isTerminated;

    /**
     * Constructs an encoder.
     */
    Encoder(CharsetEncoder csEncoder, CharBuffer charBuffer, ByteBuffer byteBuffer) {
        this.csEncoder = csEncoder
            .onMalformedInput(CodingErrorAction.REPLACE)
            .onUnmappableCharacter(CodingErrorAction.REPLACE);
        this.charBuffer = charBuffer;
        this.byteBuffer = byteBuffer;
    }

    /**
     * Returns a new encoder for the charset.
     *
     * @param csEncoder the charset encoder.
     * @return an encoder.
     */
    public static Encoder of(CharsetEncoder csEncoder) {
        CharBuffer charBuffer = CharBuffer.wrap(new char[1024]);
        int bbSize = charBuffer.capacity() * ((int) Math.ceil(csEncoder.maxBytesPerChar()));
        ByteBuffer byteBuffer = ByteBuffer.allocate(bbSize);
        return new Encoder(csEncoder, charBuffer, byteBuffer);
    }

    /**
     * Encodes a str.
     *
     * @param str a str.
     * @return the encoded bytes.
     */
    public byte[] encode(String str) {
        byte[] bytes = new byte[0];
        CharSequence seq = str;
        while (seq.length() > 0) {
            int size = Math.min(seq.length(), this.charBuffer.remaining());
            this.charBuffer.append(seq.subSequence(0, size));
            this.charBuffer.flip();
            this.csEncoder.encode(charBuffer, byteBuffer, false);
            this.charBuffer.compact();

            this.byteBuffer.flip();
            byte[] newBytes = new byte[bytes.length + this.byteBuffer.limit()];
            System.arraycopy(bytes, 0, newBytes,0, bytes.length);
            this.byteBuffer.get(newBytes, bytes.length, this.byteBuffer.limit());
            this.byteBuffer.clear();

            bytes = newBytes;
            seq = seq.subSequence(size, seq.length());
        }
        return bytes;
    }

    /**
     * Terminates the encoder.
     *
     * Calling this method twice is not permitted.
     *
     * @return the last bytes.
     */
    public byte[] terminate() {
        this.isTerminated = true;
        this.csEncoder.encode(CharBuffer.wrap(new char[0]), byteBuffer, true);
        this.csEncoder.flush(byteBuffer);

        this.byteBuffer.flip();
        byte[] bytes = new byte[this.byteBuffer.limit()];
        this.byteBuffer.get(bytes);
        this.byteBuffer.clear();

        return bytes;
    }

    /**
     * Returns true if the encoder is already terminated.
     *
     * @return true if the encoder is already terminated.
     */
    boolean isTerminated() {
        return this.isTerminated;
    }

}

// vim: et sw=4 sts=4 fdm=marker
