package org.kink_lang.kink;

import java.util.List;
import java.util.Random;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;

import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.internal.mod.random.PrngVal;
import org.kink_lang.kink.internal.mod.random.RngVal;
import org.kink_lang.kink.internal.mod.random.Xoshiro256StarStar;
import org.kink_lang.kink.internal.str.Decoder;
import org.kink_lang.kink.internal.str.Encoder;

/**
 * Factory of funs of kink/_INTERNAL mod.
 */
class InternalMod {

    /** The vm. */
    private final Vm vm;

    /** The sym handle of "from_each". */
    private final int fromEachHandle;

    /**
     * Constructs the factory.
     */
    InternalMod(Vm vm) {
        this.vm = vm;
        this.fromEachHandle = vm.sym.handleFor("from_each");
    }

    /**
     * Makes kink/KONT_TAG mod.
     */
    Val makeMod() {
        Val mod = vm.newVal();
        mod.setVar(vm.sym.handleFor("new_prng"),
                vm.fun.make("_INTERNAL.new_prng(S0 S1 S2 S3)").take(4).action(this::newPrngFun));
        mod.setVar(vm.sym.handleFor("prng?"),
                vm.fun.make("_INTERNAL.prng?(Val)").take(1).action(this::isPrngFun));
        mod.setVar(vm.sym.handleFor("new_rng"),
                vm.fun.make("_INTERNAL.new_rng?(Random Type_tag)").take(2).action(this::newRngFun));
        mod.setVar(vm.sym.handleFor("new_encoder_pair"),
                vm.fun.make("_INTERNAL.new_encoder_pair(Cs_encoder)")
                .take(1).action(this::newEncoderPair));
        mod.setVar(vm.sym.handleFor("new_decoder_triple"),
                vm.fun.make("_INTERNAL.new_decoder_triple(Charset)")
                .take(1).action(this::newDecoderTriple));
        mod.setVar(vm.sym.handleFor("val_id"),
                vm.fun.make("VAL.val_id(Val)")
                .take(1).action(c -> vm.num.of(c.arg(0).identityDecimal())));
        mod.setVar(vm.sym.handleFor("var_syms"),
                vm.fun.make("VAL.var_syms(Val)")
                .take(1).action(this::varSyms));
        mod.setVar(vm.sym.handleFor("current_traces"),
                vm.fun.make("TRACE.current_traces")
                .take(0).action(c -> vm.vec.of(c.traces())));
        mod.setVar(vm.sym.handleFor("require"),
                vm.fun.make("MOD.require(Mod_name ...[$config])")
                .takeMinMax(1, 2).action(this::requireFun));
        return mod;
    }

    /**
     * Implementation of _INTERNAL.new_prng.
     */
    private HostResult newPrngFun(CallContext c) {
        long s0 = (Long) ((JavaVal) c.arg(0)).objectReference();
        long s1 = (Long) ((JavaVal) c.arg(1)).objectReference();
        long s2 = (Long) ((JavaVal) c.arg(2)).objectReference();
        long s3 = (Long) ((JavaVal) c.arg(3)).objectReference();
        Xoshiro256StarStar random = new Xoshiro256StarStar(s0, s1, s2, s3);
        return PrngVal.of(vm, random);
    }

    /**
     * Implementation of _INTERNAL.prng?.
     */
    private HostResult isPrngFun(CallContext c) {
        return vm.bool.of(c.arg(0) instanceof PrngVal);
    }

    /**
     * Implementation of _INTERNAL.new_rng.
     */
    private HostResult newRngFun(CallContext c) {
        Random random = (Random) ((JavaVal) c.arg(0)).objectReference();
        String typeTag = ((StrVal) c.arg(1)).string();
        return RngVal.of(vm, random, typeTag);
    }

    /**
     * Implementation of _INTERNAL.new_encoder_pair.
     */
    private HostResult newEncoderPair(CallContext c) {
        var csEncoder = (CharsetEncoder) ((JavaVal) c.arg(0)).objectReference();
        Encoder encoder = Encoder.of(csEncoder);
        var encode = vm.fun.make().take(1).action(cc -> {
            var str = ((StrVal) cc.arg(0)).string();
            return vm.bin.of(encoder.encode(str));
        });
        var terminate = vm.fun.make().take(0).action(cc -> {
            return vm.bin.of(encoder.terminate());
        });
        return vm.vec.of(encode, terminate);
    }

    /**
     * Implementation of _INTERNAL.new_decoder_triple.
     */
    private HostResult newDecoderTriple(CallContext c) {
        var charset = (Charset) ((JavaVal) c.arg(0)).objectReference();
        Decoder decoder = Decoder.of(charset);
        var consume = vm.fun.make().take(1).action(cc -> {
            var bin = (BinVal) cc.arg(0);
            decoder.consume(bin.bytes());
            return vm.nada;
        });
        var terminate = vm.fun.make().take(0).action(cc -> {
            decoder.terminate();
            return vm.nada;
        });
        var emitText = vm.fun.make().take(0).action(cc -> vm.str.of(decoder.emitText()));
        return vm.vec.of(consume, terminate, emitText);
    }

    /**
     * Implementation of VAL.var_syms.
     */
    private HostResult varSyms(CallContext c) {
        var val = c.arg(0);
        List<StrVal> syms = val.getVarSymHandleSet().stream()
            .map(handle -> vm.str.of(vm.sym.symFor(handle)))
            .toList();
        return c.call("kink/container/FLAT_SET", fromEachHandle).args(vm.vec.of(syms));
    }

    // require {{{1

    /**
     *  Implementation of require.
     */
    private HostResult requireFun(CallContext c) throws Throwable {
        Val modNameVal = c.arg(0);
        if (! (modNameVal instanceof StrVal)) {
            return c.call(vm.graph.raiseFormat(
                        "require(Mod_name ...[$config]): Mod_name must be a str, but got {}",
                        vm.graph.repr(modNameVal)));
        }
        String modName = ((StrVal) modNameVal).string();

        if (c.argCount() == 1) {
            return vm.mod.require(c, modName, null, null, null);
        } else {
            Val config = c.arg(1);
            FunVal cont = vm.fun.make().take(3).action(
                    cc -> requireWithConts(cc, modName, cc.arg(0), cc.arg(1), cc.arg(2)));
            int readConf = vm.sym.handleFor("read_config");
            return c.call("kink/_mod/REQUIRE_AUX", readConf).args(modNameVal, config, cont);
        }
    }

    /**
     * Continues to the body of require, with the given conts.
     */
    HostResult requireWithConts(
            HostContext c, String modName, Val successCont, Val notFoundCont, Val compileErrorCont
            ) throws Throwable {
        if (! (successCont instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat("require(Mod_name ...[$config]): "
                        + "success cont is not a fun but {}",
                        vm.graph.repr(successCont)));
        }

        if (! (notFoundCont instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat("require(Mod_name ...[$config]): "
                        + "not found cont is not a fun but {}",
                        vm.graph.repr(notFoundCont)));
        }

        if (! (compileErrorCont instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat("require(Mod_name ...[$config]): "
                        + "compile error cont is not a fun but {}",
                        vm.graph.repr(compileErrorCont)));
        }

        return vm.mod.require(c, modName,
                (FunVal) successCont,
                (FunVal) notFoundCont,
                (FunVal) compileErrorCont);
    }

    // }}}1

}

// vim: et sw=4 sts=4 fdm=marker
