package org.kink_lang.kink;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.function.ThrowingFunction0;
import org.kink_lang.kink.internal.function.ThrowingFunction1;
import org.kink_lang.kink.internal.function.ThrowingFunction3;
import org.kink_lang.kink.hostfun.HostFunBuilder;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.hostfun.graph.GraphNode;
import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.internal.num.NumOperations;

/**
 * The helper of {@link JavaVal}s.
 */
public class JavaHelper {

    /** The vm. */
    private final Vm vm;

    /** Shared vars of java vals.. */
    SharedVars sharedVars;

    /** The handle of "read_config". */
    private int readConfigHandle;

    /** The default success cont of invoecation. */
    private FunVal invDefSuccessCont;

    /** The default error cont of invoecation. */
    private FunVal invDefErrorCont;

    /**
     * Makes the helper.
     */
    JavaHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Returns a java val.
     *
     * <p>The {@code object} must be able to typed as {@code staticType}.
     * Namely,</p>
     *
     * <ul>
     *   <li>{@code object} is null and {@code staticType} is not a primitive type,</li>
     *   <li>{@code object} is a boxed value and {@code staticType}
     *   is the corresponding primitive type,</li>
     *   <li>or {@code object} is not null and {@code staticType.isInstance(object)} is true.</li>
     * </ul>
     *
     * @param object the nullable java object reference
     * @param staticType the static type.
     * @return a java val.
     * @throws IllegalArgumentException if {@code object} cannot be typed as {@code staticType}.
     */
    public JavaVal of(@Nullable Object object, Class<?> staticType) {
        return new JavaVal(vm, object, staticType);
    }

    /**
     * Intializes the helper.
     */
    void init() {
        Map<Integer, Val> map = new HashMap<>();
        addMethod(map, "Java_val", "static_type", "", 0, this::staticTypeFun);
        addMethod(map, "Java_val", "dynamic_type", "", 0, this::dynamicTypeFun);
        addMethod(map, "Java_val", "typable_as?", "(Klass)", 1, this::typableAsFun);
        addMethod(map, "Java_val", "as", "(Klass)", 1, this::asFun);
        addMethod(map, "Java_val", "eq_eq?", "(Another)", 1, this::eqEqPFun);
        addMethod(map, "Java_val", "null?", "", 0,
                (c, v, d) -> vm.bool.of(v.objectReference() == null));
        addMethod(map, "Java_val", "to_kink_str", "", 0, this::toKinkStrFun);
        addMethod(map, "Java_val", "to_kink_bool", "", 0, this::toKinkBoolFun);
        addMethod(map, "Java_val", "to_kink_num", "", 0, this::toKinkNumFun);
        addMethod(map, "Java_val", "to_kink_bin", "(...[From=0 To=size])",
                c -> c.takeMinMax(0, 2), this::toKinkBinFun);
        addMethod(map, "Java_val", "to_kink_exception", "", 0, this::toKinkExceptionMethod);
        addMethod(map, "Java_val", "copy_from_bin", "(At Bin From To)",
                c -> c.take(4), this::copyFromBIn);
        addMethod(map, "Java_val", "unwrap", "", 0, this::unwrapFun);
        addMethod(map, "Java_val", "array_class", "", 0, this::arrayClassFun);
        addMethod(map, "Java_val", "array_new", "(Size)", 1, this::arrayNewFun);
        addMethod(map, "Java_val", "array_of", "(...[E0 E1 ,,,])", c -> c, this::arrayOfFun);
        addMethod(map, "Java_val", "array_length", "", 0, this::arrayLengthFun);
        addMethod(map, "Java_val", "array_get", "(Ind)", 1, this::arrayGetFun);
        addMethod(map, "Java_val", "array_set", "(Ind Elem)", 2, this::arraySetFun);
        addMethod(map, "Java_val", "get_field", "(Field_name)", 1, this::getFieldFun);
        addMethod(map, "Java_val", "set_field", "(Field_name Content)", 2, this::setFieldFun);
        addMethod(map, "Java_val", "get_static", "(Field_name)", 1, this::getStaticFun);
        addMethod(map, "Java_val", "set_static", "(Field_name Content)", 2, this::setStaticFun);
        addMethod(map, "Java_val", "call_method", "(Method_name ...[A0 A1 ,,,] ...[$config])",
                c -> c.takeMin(1), this::callMethodFun);
        addMethod(map, "Java_val", "call_static", "(Method_name ...[A0 A1 ,,,] ...[$config])",
                c -> c.takeMin(1), this::callStaticFun);
        addMethod(map, "Java_val", "new", "(...[A0 A1 ,,,] ...[$config])", c -> c, this::newFun);
        addMethod(map, "Java_val", "repr", "", 0, this::reprFun);

        this.sharedVars = vm.sharedVars.of(map);
        this.readConfigHandle = vm.sym.handleFor("read_config");
        this.invDefSuccessCont = vm.fun.make().takeMinMax(0, 1).action(this::invDefSuccessCont);
        this.invDefErrorCont = vm.fun.make().take(1).action(this::invDefErrorCont);
    }

    // Java_val.static_type {{{1

    /**
     * Implementation of Java_val.static_type.
     */
    private HostResult staticTypeFun(CallContext c, JavaVal jv, String desc) {
        return vm.java.of(jv.staticType(), Class.class);
    }

    // }}}1

    // Java_val.dynamic_type {{{1

    /**
     * Implementation of Java_val.dynamic_type.
     */
    private HostResult dynamicTypeFun(CallContext c, JavaVal jv, String desc) {
        Object obj = jv.objectReference();
        if (obj == null) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: Java_val must be non-null java val, but got null", desc));
        }
        return vm.java.of(obj.getClass(), Class.class);
    }

    // }}}1

    // Java_val.typable_as? {{{1

    /**
     * Implementation of Java_val.typable_as?.
     */
    private HostResult typableAsFun(CallContext c, JavaVal jv, String desc) {
        Val klassVal = c.arg(0);
        Class<?> klass = asClass(klassVal);
        if (klass == null) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Klass must be a Java class, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(klassVal)));
        }
        return vm.bool.of(isTypable(jv.objectReference(), klass));
    }

    // }}}1

    // Java_val.as {{{1

    /**
     * Implementation of Java_val.as.
     */
    private HostResult asFun(CallContext c, JavaVal jv, String desc) {
        Val klassVal = c.arg(0);
        Class<?> klass = asClass(klassVal);
        if (klass == null) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Klass must be a Java class, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(klassVal)));
        }

        Object obj = jv.objectReference();
        if (! isTypable(obj, klass)) {
            String msg = jv.objectReference() == null
                ? String.format(Locale.ROOT,
                        "%s: null is not typable as %s", desc, klass.getName())
                : String.format(Locale.ROOT,
                        "%s: instance of %s is not typable as %s",
                        desc,
                        obj.getClass().getName(), klass.getName());
            return c.raise(msg);
        }
        return vm.java.of(jv.objectReference(), klass);
    }

    // }}}1

    // Java_val.eq_eq? {{{1

    /**
     * Implementation of Java_val.eq_eq?.
     */
    private HostResult eqEqPFun(CallContext c, JavaVal jv, String desc) {
        if (! (c.arg(0) instanceof JavaVal argJval)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Another must be a java val, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        return vm.bool.of(jv.objectReference() == argJval.objectReference());
    }

    // }}}1

    // Java_val.to_kink_str {{{1

    /**
     * Implementation of Java_val.to_kink_str.
     */
    private HostResult toKinkStrFun(CallContext c, JavaVal jv, String desc) {
        if (! (jv.objectReference() instanceof String str)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Java_val must be a java val of String, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(jv)));
        }
        return vm.str.of(str);
    }

    // }}}1

    // Java_val.to_kink_bool {{{1

    /**
     * Implementation of Java_val.to_kink_bool.
     */
    private HostResult toKinkBoolFun(CallContext c, JavaVal jv, String desc) {
        if (! (jv.objectReference() instanceof Boolean bl)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Java_val must be a java val of Boolean, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(jv)));
        }
        return vm.bool.of(bl);
    }

    // }}}1

    // Java_val.to_kink_num {{{1

    /**
     * Implementation of Java_val.to_kink_num.
     */
    private HostResult toKinkNumFun(CallContext c, JavaVal jv, String desc) {
        Object obj = jv.objectReference();

        if (obj instanceof Character) {
            return vm.num.of((Character) obj);
        } else if (obj instanceof Byte) {
            return vm.num.of((Byte) obj);
        } else if (obj instanceof Short) {
            return vm.num.of((Short) obj);
        } else if (obj instanceof Integer) {
            return vm.num.of((Integer) obj);
        } else if (obj instanceof Long) {
            return vm.num.of((Long) obj);
        } else if (obj instanceof Float) {
            float num = (Float) obj;
            if (! Float.isFinite(num)) {
                return c.raise(String.format(Locale.ROOT,
                            "Java_val.to_kink_num: cannot convert %s to kink num", num));
            }
            @SuppressWarnings("PMD.AvoidDecimalLiteralsInBigDecimalConstructor")
            var r = vm.num.of(new BigDecimal((double) num));
            return r;
        } else if (obj instanceof Double) {
            double num = (Double) obj;
            if (! Double.isFinite(num)) {
                return c.raise(String.format(Locale.ROOT,
                            "Java_val.to_kink_num: cannot convert %s to kink num", num));
            }
            @SuppressWarnings("PMD.AvoidDecimalLiteralsInBigDecimalConstructor")
            var r = vm.num.of(new BigDecimal((Double) obj));
            return r;
        } else if (obj instanceof BigInteger) {
            return vm.num.of((BigInteger) obj);
        } else if (obj instanceof BigDecimal) {
            return vm.num.of((BigDecimal) obj);
        }

        return c.call(vm.graph.raiseFormat(
                    "Java_val.to_kink_num: required char, byte, short, int, long,"
                    + " finite float, finite double, BigInteger or BigDecimal as \\recv,"
                    + " but got {}",
                    vm.graph.repr(jv)));
    }

    // }}}1

    // Java_val.to_kink_bin {{{1

    /**
     * Implementation of Java_val.to_kink_bin.
     */
    private HostResult toKinkBinFun(CallContext c, JavaVal jv, String desc) {
        if (! (jv.objectReference() instanceof byte[] bytes)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Java_val must be java val of byte[], but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(jv)));
        }

        if (c.argCount() == 0) {
            return vm.bin.of(bytes);
        }

        Val fromVal = c.arg(0);
        if (! (fromVal instanceof NumVal fromNum)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: From must be a num, but was {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(fromVal)));
        }
        BigDecimal from = fromNum.bigDecimal();

        BigDecimal to;
        if (c.argCount() == 1) {
            to = BigDecimal.valueOf(bytes.length);
        } else {
            Val toVal = c.arg(1);
            if (! (toVal instanceof NumVal toNum)) {
                return c.call(vm.graph.raiseFormat(
                            "{}: To must be a num, but was {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(toVal)));
            }
            to = toNum.bigDecimal();
        }

        if (! NumOperations.isRangePair(from, to, bytes.length)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: [From, To] must be in [0, {}], but was [{}, {}]",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(vm.num.of(bytes.length)),
                        vm.graph.repr(fromVal),
                        vm.graph.repr(vm.num.of(to))));
        }

        return vm.bin.of(bytes, from.intValue(), to.intValue());
    }

    // }}}1

    // Java_val.to_kink_exception {{{

    /**
     * Implementation of Java_val.to_kink_exception.
     */
    private HostResult toKinkExceptionMethod(CallContext c, JavaVal recv, String desc) {
        if (! (recv.objectReference() instanceof Throwable th)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Java_val must be java val of Throwable, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(recv)));
        }
        return vm.exception.of(th);
    }

    // }}}

    // Java_val.copy_from_bin {{{

    /**
     * Implementation of {@code Java_val.copy_from_bin(At Bin From To)}.
     */
    private HostResult copyFromBIn(CallContext c, JavaVal recv, String desc) {
        if (! (recv.objectReference() instanceof byte[] bytes)) {
            return c.call(vm.graph.raiseFormat("{}: Java_val must be byte[], but was {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(recv)));
        }

        if (! (c.arg(0) instanceof NumVal atNum)) {
            return c.call(vm.graph.raiseFormat("{}: At must be a num, but was {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        BigDecimal at = atNum.bigDecimal();

        if (! (c.arg(1) instanceof BinVal bin)) {
            return c.call(vm.graph.raiseFormat("{}: Bin must be a bin, but was {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(1))));
        }

        if (! (c.arg(2) instanceof NumVal fromNum)) {
            return c.call(vm.graph.raiseFormat("{}: From must be a bin, but was {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(2))));
        }
        BigDecimal from = fromNum.bigDecimal();

        if (! (c.arg(3) instanceof NumVal toNum)) {
            return c.call(vm.graph.raiseFormat("{}: To must be a bin, but was {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(3))));
        }
        BigDecimal to = toNum.bigDecimal();

        if (! NumOperations.isRangePair(from, to, bin.size())) {
            return c.call(vm.graph.raiseFormat(
                        "{}: required 0 <= From <= To <= {}, but From={} To={}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(bin.size())),
                        vm.graph.repr(fromNum),
                        vm.graph.repr(toNum)));
        }
        int fromPos = from.intValue();
        int toPos = to.intValue();

        int atPos = NumOperations.getPosIndex(at, bytes.length);
        if (atPos < 0) {
            return c.call(vm.graph.raiseFormat("{}: At must be a pos index in byte[{}], but was {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(bytes.length)),
                        vm.graph.repr(atNum)));
        }

        int copySize = toPos - fromPos;
        int destEndPos = atPos + copySize;
        if (destEndPos > bytes.length) {
            return c.call(vm.graph.raiseFormat(
                        "{}: From={} To={} At={} make the destination end pos {},"
                        + " which is out of bound of byte[{}]",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(fromNum),
                        vm.graph.of(toNum),
                        vm.graph.of(atNum),
                        vm.graph.of(vm.num.of(destEndPos)),
                        vm.graph.of(vm.num.of(bytes.length))));
        }

        bin.copyToBytes(fromPos, toPos, bytes, atPos);
        return vm.nada;
    } // }}}

    // Java_val.unwrap {{{1

    /**
     * Implementation of Java_val.unwrap.
     */
    private HostResult unwrapFun(HostContext c, JavaVal jv, String desc) {
        if (! (jv.objectReference() instanceof Val unwrapped)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Java_val must be a java val of org.kink_lang.kink.Val, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(jv)));
        }

        if (unwrapped.vm != this.vm) {
            return c.raise(desc + ": could not unwrap val from another Kink VM");
        }
        return unwrapped;
    }

    // }}}1

    // Java_val.array_class {{{1

    /**
     * Implementation of Java_val.array_class.
     */
    private HostResult arrayClassFun(CallContext c, JavaVal klassVal, String desc) {
        Object klassObj = klassVal.objectReference();
        Optional<String> errorMsg = checkErrorForArrayComponentClass(desc, klassObj);
        return errorMsg.map(c::raise).orElseGet(() -> {
            Object array = Array.newInstance((Class<?>) klassObj, 0);
            return vm.java.of(array.getClass(), Class.class);
        });
    }

    // }}}1

    // Java_val.array_new(Size) {{{1

    /**
     * Implementation of Java_val.array_new(Size).
     */
    private HostResult arrayNewFun(CallContext c, JavaVal klassVal, String desc) {
        Object klassObj = klassVal.objectReference();
        Optional<String> errorMsg = checkErrorForArrayComponentClass(desc, klassObj);
        if (errorMsg.isPresent()) {
            return c.raise(errorMsg.get());
        }
        Val sizeVal = c.arg(0);
        int size = NumOperations.getPosIndex(sizeVal, Integer.MAX_VALUE);
        if (size < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Size must be an int num in [0, {}], but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(Integer.MAX_VALUE)),
                        vm.graph.repr(sizeVal)));
        }
        Object array = Array.newInstance((Class<?>) klassObj, size);
        return vm.java.of(array, array.getClass());
    }

    // }}}1

    // Java_val.array_of(E0 E1 ...) {{{1

    /**
     * Implementation of Java_val.array_of(E0 E1 ...).
     */
    private HostResult arrayOfFun(CallContext c, JavaVal klassVal, String desc) {
        Object klassObj = klassVal.objectReference();
        Optional<String> errorMsg = checkErrorForArrayComponentClass(desc, klassObj);
        if (errorMsg.isPresent()) {
            return c.raise(errorMsg.get());
        }
        Class<?> klass = (Class<?>) klassObj;
        Object array = Array.newInstance(klass, c.argCount());
        for (int i = 0; i < c.argCount(); ++ i) {
            Val elem = c.arg(i);
            if (! (elem instanceof JavaVal)) {
                return c.call(vm.graph.raiseFormat(
                            "{}: E{} must be a java val, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.num.of(i)),
                            vm.graph.repr(elem)));
            }
            Object elemObj = ((JavaVal) elem).objectReference();
            if (! isTypable(elemObj, klass)) {
                return c.call(vm.graph.raiseFormat(
                            "{}: E{} must be a java val typable as {}, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(klass.getName())),
                            vm.graph.of(vm.num.of(i)),
                            vm.graph.repr(elem)));
            }
            Array.set(array, i, elemObj);
        }
        return vm.java.of(array, array.getClass());
    }

    // }}}1

    // Java_val.array_length {{{1

    /**
     * Implementation of Java_val.array_length.
     */
    private HostResult arrayLengthFun(CallContext c, JavaVal jval, String desc) {
        Object obj = jval.objectReference();
        return checkErrorForArray(desc, obj)
            .map(c::raise)
            .orElseGet(() -> vm.num.of(Array.getLength(obj)));
    }

    // }}}1

    // Java_val.array_get(Index) {{{1

    /**
     * Implementation of Java_val.array_get(Index).
     */
    private HostResult arrayGetFun(CallContext c, JavaVal jv, String desc) {
        Object arrayObj = jv.objectReference();
        Optional<String> errorMsg = checkErrorForArray(desc, arrayObj);
        if (errorMsg.isPresent()) {
            return c.raise(errorMsg.get());
        }

        int size = Array.getLength(arrayObj);
        Val indexVal = c.arg(0);
        int index = NumOperations.getElemIndex(indexVal, size);
        if (index < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Ind must be an int num in [0, {}), but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indexVal)));
        }
        return vm.java.of(Array.get(arrayObj, index), arrayObj.getClass().getComponentType());
    }

    // }}}1

    // Java_val.array_set(Index Elem) {{{1

    /**
     * Implementation of Java_val.array_set(Index Elem).
     */
    private HostResult arraySetFun(CallContext c, JavaVal jv, String desc) throws Throwable {
        Object arrayObj = jv.objectReference();
        Optional<String> errorMsg = checkErrorForArray(desc, arrayObj);
        if (errorMsg.isPresent()) {
            return c.raise(errorMsg.get());
        }

        int size = Array.getLength(arrayObj);
        Val indexVal = c.arg(0);
        int index = NumOperations.getElemIndex(indexVal, size);
        if (index < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Ind must be an int num in [0, {}), but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(indexVal)));
        }

        Class<?> componentType = arrayObj.getClass().getComponentType();
        Val elemVal = c.arg(1);
        return forValOf(componentType, elemVal, () -> c.call(vm.graph.raiseFormat(
                        "{}: Elem msut be a java val typable as {}, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.str.of(componentType.getName())),
                        vm.graph.repr(elemVal))),
                elem -> {
                    Array.set(arrayObj, index, elem);
                    return vm.nada;
                });
    }

    // }}}1

    // Java_val.get_field {{{1

    /**
     * Implementation of Java_val.get_field.
     */
    private HostResult getFieldFun(
            CallContext c, JavaVal jv, String desc
            ) throws ReflectiveOperationException {
        if (jv.objectReference() == null) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: Java_val must be non-null java val, but got null",
                        desc));
        }

        if (! (c.arg(0) instanceof StrVal nameVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Field_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String name = nameVal.string();

        Field field = jv.staticType().getField(name);
        if (Modifier.isStatic(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: required instance field name, but %s.%s is static field",
                    desc,
                    jv.staticType().getName(),
                    name);
            return c.raise(msg);
        }

        Object fieldVal = field.get(jv.objectReference());
        return vm.java.of(fieldVal, field.getType());
    }

    // }}}1

    // Java_val.set_field {{{1

    /**
     * Implementation of Java_val.set_field.
     */
    private HostResult setFieldFun(
            CallContext c, JavaVal jv, String desc
            ) throws ReflectiveOperationException {
        if (jv.objectReference() == null) {
            return c.raise(String.format(Locale.ROOT,
                    "%s: Java_val must be a non-null java val, but got null",
                    desc));
        }

        if (! (c.arg(0) instanceof StrVal nameVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Field_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String name = nameVal.string();

        if (! (c.arg(1) instanceof JavaVal content)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Content must be a java val, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(1))));
        }
        Object contentObj = content.objectReference();

        Field field = jv.staticType().getField(name);
        if (Modifier.isStatic(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: required instance field name, but %s.%s is static field",
                    desc,
                    jv.staticType().getName(), name);
            return c.raise(msg);
        }
        if (Modifier.isFinal(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: cannot set a value because %s.%s is a final field",
                    desc,
                    jv.staticType().getName(),
                    name);
            return c.raise(msg);
        }

        field.set(jv.objectReference(), contentObj);
        return vm.nada;
    }

    // }}}1

    // Java_val.get_static {{{1

    /**
     * Implementation of Java_val.get_static.
     */
    private HostResult getStaticFun(
            CallContext c, JavaVal jv, String desc
            ) throws ReflectiveOperationException {
        if (jv.objectReference() == null) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: Java_val must be java val of class, but got null",
                        desc));
        }

        if (! (jv.objectReference() instanceof Class)) {
            Class<?> dynamicClass = jv.objectReference().getClass();
            String msg = String.format(Locale.ROOT,
                    "%s: Java_val must be java val of class, but got %s: %s",
                    desc,
                    dynamicClass.getName(),
                    jv.objectReference());
            return c.raise(msg);
        }
        Class<?> klass = (Class<?>) jv.objectReference();

        if (! (c.arg(0) instanceof StrVal nameVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Field_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String name = nameVal.string();

        Field field = klass.getField(name);
        if (! Modifier.isStatic(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: required static field name, but %s.%s is instance field",
                    desc,
                    klass.getName(),
                    name);
            return c.raise(msg);
        }

        Object fieldVal = field.get(null);
        return vm.java.of(fieldVal, field.getType());
    }

    // }}}1

    // Java_val.set_static {{{1

    /**
     * Implementation of Java_val.set_static.
     */
    private HostResult setStaticFun(
            CallContext c, JavaVal jv, String desc
            ) throws ReflectiveOperationException {
        Class<?> klass = asClass(jv);
        if (klass == null) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Java_val must be a java val of class, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(jv)));
        }

        if (! (c.arg(0) instanceof StrVal nameVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Field_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String name = nameVal.string();

        Field field = klass.getField(name);
        if (! Modifier.isStatic(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: required static field name, but %s.%s is instance field",
                    desc,
                    klass.getName(),
                    name);
            return c.raise(msg);
        }
        if (Modifier.isFinal(field.getModifiers())) {
            String msg = String.format(Locale.ROOT,
                    "%s: cannot set a value because %s.%s is a final field",
                    desc,
                    klass.getName(),
                    name);
            return c.raise(msg);
        }

        if (! (c.arg(1) instanceof JavaVal content)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Content must be a java val, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(1))));
        }
        Object contentObj = content.objectReference();

        field.set(null, contentObj);
        return vm.nada;
    }

    // }}}1

    // Java_val.call_method, call_static, new {{{1

    /**
     * Implementation of Java_val.call_method.
     */
    private HostResult callMethodFun(
            CallContext c, JavaVal recv, String desc) throws Throwable {
        if (! (c.arg(0) instanceof StrVal nameVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Method_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String name = nameVal.string();

        Optional<FunVal> config = retrieveConfig(c);
        var maybeArgs = retrieveArgs(c, 1, config.isPresent(), desc);

        if (maybeArgs instanceof ArgsError error) {
            return c.call(error.error());
        }
        var args = (Args) maybeArgs;

        Method method;
        try {
            method = recv.staticType().getMethod(name, args.argTypes());
        } catch (NoSuchMethodException nsme) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: method not found: %s",
                        desc,
                        nsme.getMessage()));
        }
        if (Modifier.isStatic(method.getModifiers())) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: method %s is a static method",
                        desc,
                        name));
        }
        return withConts(c, config, (cc, successCont, errorCont) -> {
            return invokeMethod(
                    cc, method, recv.objectReference(), args.args(), successCont, errorCont);
        });
    }

    /**
     * Implementation of Java_val.call_static.
     */
    private HostResult callStaticFun(
            CallContext c, JavaVal klassVal, String desc) throws Throwable {
        if (! (klassVal.objectReference() instanceof Class<?> klass)) {
            String msg = String.format(Locale.ROOT,
                    "%s: Java_val must be a java val of class, but got %s",
                    desc,
                    objectDesc(klassVal.objectReference()));
            return c.raise(msg);
        }

        if (! (c.arg(0) instanceof StrVal nameVal)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Method_name must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String name = nameVal.string();

        Optional<FunVal> config = retrieveConfig(c);
        var maybeArgs = retrieveArgs(c, 1, config.isPresent(), desc);

        if (maybeArgs instanceof ArgsError error) {
            return c.call(error.error());
        }
        var args = (Args) maybeArgs;

        Method method;
        try {
            method = klass.getMethod(name, args.argTypes());
        } catch (NoSuchMethodException nsme) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: method not found: %s",
                        desc,
                        nsme.getMessage()));
        }
        if (! Modifier.isStatic(method.getModifiers())) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: method %s is not a static method",
                        desc,
                        name));
        }
        return withConts(c, config, (cc, successCont, errorCont) -> {
            return invokeMethod(cc, method, null, args.args(), successCont, errorCont);
        });
    }

    /**
     * Implementation of Java_val.new.
     */
    private HostResult newFun(CallContext c, JavaVal klassVal, String desc) throws Throwable {
        if (! (klassVal.objectReference() instanceof Class<?> klass)) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: Java_val must be a java val of class, but got %s",
                        desc,
                        objectDesc(klassVal.objectReference())));
        }

        Optional<FunVal> config = retrieveConfig(c);
        var maybeArgs = retrieveArgs(c, 0, config.isPresent(), desc);

        if (maybeArgs instanceof ArgsError error) {
            return c.call(error.error());
        }
        var args = (Args) maybeArgs;

        Constructor ctor;
        try {
            ctor = klass.getConstructor(args.argTypes());
        } catch (NoSuchMethodException nsme) {
            return c.raise(String.format(Locale.ROOT,
                        "%s: constructor not found: %s", desc, nsme.getMessage()));
        }
        return withConts(c, config, (cc, successCont, errorCont) -> {
            return invokeConstructor(cc, ctor, args.args(), successCont, errorCont);
        });
    }

    /**
     * Retrieves the configu fun of new, call_method, call_static, if any.
     */
    private Optional<FunVal> retrieveConfig(CallContext c) {
        int argCount = c.argCount();
        return argCount != 0 && c.arg(argCount - 1) instanceof FunVal config
            ? Optional.of(config)
            : Optional.empty();
    }

    /**
     * Maybe args of invocation.
     */
    private sealed interface MaybeArgs {}

    /**
     * Success result of args retrieval.
     *
     * @param argTypes the static types of args.
     * @param args the args of invocation.
     */
    private record Args(Class<?>[] argTypes, Object[] args) implements MaybeArgs {}

    /**
     * Error result of args retrieval.
     *
     * @param error the error result of the caller of invocation.
     */
    private record ArgsError(GraphNode error) implements MaybeArgs {}

    /**
     * Retrieves args of invocation.
     */
    private MaybeArgs retrieveArgs(
            CallContext c, int offset, boolean hasConfig, String desc) {
        int arity = c.argCount() - offset - (hasConfig ? 1 : 0);
        Class<?>[] argTypes = new Class[arity];
        Object[] args = new Object[arity];
        for (int i = 0; i < arity; ++ i) {
            Val arg = c.arg(offset + i);
            if (! (arg instanceof JavaVal argJava)) {
                return new ArgsError(vm.graph.raiseFormat(
                            "{}: A{} must be a java val, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.num.of(i)),
                            vm.graph.repr(arg)));
            }
            argTypes[i] = argJava.staticType();
            args[i] = argJava.objectReference();
        }
        return new Args(argTypes, args);
    }

    /**
     * Invokes the constructor.
     */
    private HostResult invokeConstructor(
            CallContext c,
            Constructor ctor,
            Object[] args,
            @Nullable FunVal successCont,
            @Nullable FunVal errorCont
            ) {
        Object result;
        try {
            result = ctor.newInstance(args);
        } catch (ReflectiveOperationException roe) {
            Throwable th = unwrapReflectionException(roe);
            return errorCont != null
                ? c.call(errorCont).args(vm.java.of(th, Throwable.class))
                : c.raise(th);
        }
        JavaVal resultJavaVal = vm.java.of(result, ctor.getDeclaringClass());
        return successCont != null
            ? c.call(successCont).args(resultJavaVal)
            : resultJavaVal;
    }

    /**
     * Invokes the method.
     */
    private HostResult invokeMethod(
            HostContext c,
            Method method,
            Object recv,
            Object[] args,
            @Nullable FunVal successCont,
            @Nullable FunVal errorCont) {
        Class<?> returnType = method.getReturnType();
        Object result;
        try {
            result = method.invoke(recv, args);
        } catch (ReflectiveOperationException roe) {
            Throwable th = unwrapReflectionException(roe);
            return errorCont != null
                ? c.call(errorCont).args(vm.java.of(th, Throwable.class))
                : c.raise(th);
        }
        if (successCont == null) {
            return returnType.equals(void.class)
                ? vm.nada
                : vm.java.of(result, returnType);
        } else if (returnType.equals(void.class)) {
            return c.call(successCont).args();
        } else {
            Val resultVal = vm.java.of(result, returnType);
            return c.call(successCont).args(resultVal);
        }
    }

    /**
     * Invokes invoke with success and error conts.
     */
    private HostResult withConts(
            CallContext c,
            Optional<FunVal> config,
            ThrowingFunction3<CallContext, FunVal, FunVal, HostResult> invoke
            ) throws Throwable {
        if (config.isEmpty()) {
            return invoke.apply(c, null, null);
        }
        FunVal cont = vm.fun.make().take(2).action(cc -> contToInvoke(cc, invoke));
        return c.call("kink/_java/CALL_AUX", readConfigHandle)
            .args(config.get(), this.invDefSuccessCont, this.invDefErrorCont, cont);
    }

    /**
     * Continues to invoke, checking errors.
     */
    HostResult contToInvoke(
            CallContext c,
            ThrowingFunction3<CallContext, FunVal, FunVal, HostResult> invoke
            ) throws Throwable {
        Val successContVal = c.arg(0);
        if (! (successContVal instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "(cont-to-invoke($success_cont $error_cont)):"
                        + " $success_cont must be a fun, but got {}",
                        vm.graph.repr(successContVal)));
        }

        Val errorContVal = c.arg(1);
        if (! (errorContVal instanceof FunVal)) {
            return c.call(vm.graph.raiseFormat(
                        "(cont-to-invoke($success_cont $error_cont)):"
                        + " $error_cont must be a fun, but got {}",
                        vm.graph.repr(errorContVal)));
        }

        return invoke.apply(c, (FunVal) successContVal, (FunVal) errorContVal);
    }

    /**
     * Implementation of the default success cont of invocation.
     */
    HostResult invDefSuccessCont(CallContext c) {
        return c.argCount() == 0
            ? vm.nada
            : c.arg(0);
    }

    /**
     * Implementation of the default error cont of invocation.
     */
    HostResult invDefErrorCont(CallContext c) {
        Val exVal = c.arg(0);
        Supplier<HostResult> raise = () -> c.call(vm.graph.raiseFormat(
                    "(default-error-cont(Exc_java_val)):"
                    + " Exc_java_val must be a java val of Throwable, but got {}",
                    vm.graph.repr(exVal)));

        if (! (exVal instanceof JavaVal)) {
            return raise.get();
        }

        Object thObj = ((JavaVal) exVal).objectReference();
        if (! (thObj instanceof Throwable)) {
            return raise.get();
        }

        return c.raise((Throwable) thObj);
    }

    /**
     * Unwrap causing exception in the invocation target if any.
     */
    static Throwable unwrapReflectionException(ReflectiveOperationException roe) {
        if (roe instanceof InvocationTargetException) {
            Throwable cause = roe.getCause();
            if (cause != null) {
                return cause;
            }
        }
        return roe;
    }

    // }}}1

    // Java_val.repr {{{1

    /**
     * Implementation of Java_val.repr.
     */
    private HostResult reprFun(CallContext c, JavaVal java, String desc) {
        return vm.str.of(String.format(Locale.ROOT,
                    "(java %s as %s)", java.objectReference(), java.staticType().getName()));
    }

    // }}}1

    /**
     * Continues to onOk if val is a java val of klass,
     * or to onError.
     */
    private HostResult forValOf(Class<?> klass, Val val,
                                ThrowingFunction0<HostResult> onError,
                                ThrowingFunction1<Object, HostResult> onOk) throws Throwable {
        if (! (val instanceof JavaVal)) {
            return onError.apply();
        }

        Object obj = ((JavaVal) val).objectReference();
        if (! isTypable(obj, klass)) {
            return onError.apply();
        }

        return onOk.apply(obj);
    }

    /**
     * Retrieves a class from the klassVal which is expected to be a class java val.
     */
    private Class<?> asClass(Val klassVal) {
        if (! (klassVal instanceof JavaVal)) {
            return null;
        }

        JavaVal klassJavaVal = (JavaVal) klassVal;
        Object klassObj = klassJavaVal.objectReference();
        if (! (klassObj instanceof Class)) {
            return null;
        }

        return (Class<?>) klassObj;
    }

    /**
     * Adds a fixed arity method to mapping.
     */
    private void addMethod(
            Map<Integer, Val> mapping,
            String recvDesc,
            String sym,
            String argsDesc,
            int arity,
            ThrowingFunction3<CallContext, JavaVal, String, HostResult> action) {
        addMethod(mapping, recvDesc, sym, argsDesc, c -> c.take(arity), action);
    }

    /**
     * Adds a method to mapping.
     */
    private void addMethod(
            Map<Integer, Val> mapping,
            String recvDesc,
            String sym,
            String argsDesc,
            UnaryOperator<HostFunBuilder> config,
            ThrowingFunction3<CallContext, JavaVal, String, HostResult> action) {
        String desc = String.format(Locale.ROOT, "%s.%s%s", recvDesc, sym, argsDesc);
        FunVal fun = config.apply(vm.fun.make(desc)).action(c -> {
            if (! (c.recv() instanceof JavaVal recv)) {
                return c.call(vm.graph.raiseFormat("{}: {} must be java val, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(recvDesc)),
                            vm.graph.repr(c.recv())));
            }
            return action.apply(c, recv, desc);
        });
        mapping.put(vm.sym.handleFor(sym), fun);
    }

    /**
     * Returns an error message if the klassObj is not a proper class as array components.
     */
    private Optional<String> checkErrorForArrayComponentClass(String prefix, Object klassObj) {
        if (! (klassObj instanceof Class)) {
            return Optional.of(String.format(Locale.ROOT,
                    "%s: Java_val must be a java val of class, but got %s",
                    prefix, objectDesc(klassObj)));
        }
        Class<?> klass = (Class<?>) klassObj;

        if (klass.equals(void.class)) {
            return Optional.of(String.format(Locale.ROOT,
                        "%s: no array class for void", prefix));
        }

        int dimension = arrayDimension(klass);
        if (dimension >= 255) {
            return Optional.of(String.format(Locale.ROOT,
                    "%s: too big array dimension: %d", prefix, dimension + 1));
        }

        return Optional.empty();
    }

    /**
     * Returns an error message if the obj is not an array.
     */
    private Optional<String> checkErrorForArray(String prefix, Object obj) {
        if (obj == null || ! obj.getClass().isArray()) {
            return Optional.of(String.format(Locale.ROOT,
                        "%s: Java_val must be a java val of array, but got %s",
                        prefix, objectDesc(obj)));
        } else {
            return Optional.empty();
        }
    }

    /**
     * Returns the array dimension of the class.
     */
    private static int arrayDimension(Class<?> klass) {
        int dimension = 0;
        while (klass.isArray()) {
            ++ dimension;
            klass = klass.getComponentType();
        }
        return dimension;
    }

    /**
     * Returns a description of the object reference.
     */
    private String objectDesc(@Nullable Object obj) {
        return obj == null
            ? "null"
            : String.format(Locale.ROOT, "%s: %s", obj.getClass().getName(), obj);
    }

    /**
     * Returns true if the object is typable as {@code staticType}.
     *
     * @param object the object to test.
     * @param staticType the type to test.
     * @return {@code true} if the object is typable as {@code staticType}.
     */
    public static boolean isTypable(@Nullable Object object, Class<?> staticType) {
        return object == null && ! staticType.isPrimitive()
            || staticType.isInstance(object)
            || object instanceof Boolean && staticType.equals(boolean.class)
            || object instanceof Character && staticType.equals(char.class)
            || object instanceof Byte && staticType.equals(byte.class)
            || object instanceof Short && staticType.equals(short.class)
            || object instanceof Integer && staticType.equals(int.class)
            || object instanceof Long && staticType.equals(long.class)
            || object instanceof Float && staticType.equals(float.class)
            || object instanceof Double && staticType.equals(double.class);
    }

}

// vim: et sw=4 sts=4 fdm=marker
