package org.kink_lang.kink;

import java.lang.invoke.CallSite;
import java.lang.invoke.ConstantCallSite;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.invoke.MutableCallSite;

import org.kink_lang.kink.internal.ovis.OwnVarIndexes;

/**
 * Bootstrap method of var setting operation.
 */
final class SetVarBootstrapper {

    /** The maximum index of val fields in Val class. */
    private static final int MAX_VAL_FIELD_IND = 5;

    /** Logger for the class. */
    private static final System.Logger LOGGER
        = System.getLogger(SetVarBootstrapper.class.getName());

    /** Whether always using the fallback. */
    static boolean alwaysFallback = Boolean.getBoolean(
            SetVarBootstrapper.class.getName() + ".alwaysFallback");

    static {
        LOGGER.log(System.Logger.Level.DEBUG, "alwaysFallback = {0}", alwaysFallback);
    }

    /**
     * Not instantiated.
     */
    private SetVarBootstrapper() {
    }

    /**
     * Bootstrap of var setting.
     * MethodType: (Val, Val)void.
     */
    static CallSite bootstrapSetVar(
            Lookup caller,
            String name,
            MethodType type,
            int symHandle) throws Throwable {
        if (alwaysFallback) {
            return new ConstantCallSite(fallbackMH(symHandle));
        } else {
            // Why not using MutableCallSite(MethodType):
            // There appears to be no guaratee that setTarget happens-before linking of indy.
            // https://docs.oracle.com/javase/specs/jls/se8/html/jls-17.html#jls-17.4.5
            //
            // Thus, when using MutableCallSite(MethodType) without syncAll,
            // null target might be observable from other threads.
            //
            // Here, by using MutableCallSite(MethodHandle),
            // other threads at least can see the write of fallbackMH.
            var callSite = new MutableCallSite(fallbackMH(symHandle));

            callSite.setTarget(profileMH(symHandle, callSite));
            return callSite;
        }
    }

    /**
     * MethodHandle of profiling set-var.
     */
    static MethodHandle profileMH(int symHandle, MutableCallSite callSite) throws Throwable {
        var mt = MethodType.methodType(
                void.class,
                Val.class,
                int.class,
                Val.class,
                MutableCallSite.class);
        var method = MethodHandles.lookup()
            .findStatic(SetVarBootstrapper.class, "profileSetVar", mt);
        var boundSymHandle = MethodHandles.insertArguments(method, 1, symHandle);
        return MethodHandles.insertArguments(boundSymHandle, 2, callSite);
    }

    /**
     * Do profiling and set var.
     */
    static void profileSetVar(
            Val owner,
            int symHandle,
            Val content,
            MutableCallSite callSite) throws Throwable {
        var mh = profile(owner, symHandle);
        callSite.setTarget(mh);
        mh.invoke(owner, content);
    }

    /**
     * Construct appropriate MethodHandle for symHandle.
     */
    static MethodHandle profile(Val owner, int symHandle) throws Throwable {
        var ovis = owner.getOvis();
        int varInd = ovis.getIndex(symHandle);
        return varInd >= 0
            ? cacheMH(symHandle, ovis, varInd)
            : fallbackMH(symHandle);
    }

    /**
     * MethodHandle which falls back to the non-caching method.
     */
    static MethodHandle fallbackMH(int symHandle) throws Throwable {
        MethodHandle setVarMh = MethodHandles.lookup().findVirtual(Val.class,
                "setVar",
                MethodType.methodType(void.class, int.class, Val.class));
        return MethodHandles.insertArguments(setVarMh, 1, symHandle);
    }

    /**
     * MethodHandle caching ovis.
     */
    static MethodHandle cacheMH(int symHandle, OwnVarIndexes ovis, int varInd) throws Throwable {
        return varInd <= MAX_VAL_FIELD_IND
            ? cacheValFieldMH(symHandle, ovis, varInd)
            : cacheMvMH(symHandle, ovis, varInd);
    }

    /**
     * MethodHandle caching ovis for val field.
     */
    private static MethodHandle cacheValFieldMH(
            int symHandle,
            OwnVarIndexes ovis,
            int varInd) throws Throwable {
        String methodName = "setPredictV" + varInd;
        var mt = MethodType.methodType(void.class, int.class, Val.class, OwnVarIndexes.class);
        MethodHandle method = MethodHandles.lookup().findVirtual(Val.class, methodName, mt);
        var boundSymhandle = MethodHandles.insertArguments(method, 1, symHandle);
        return MethodHandles.insertArguments(boundSymhandle, 2, ovis);
    }

    /**
     * MethodHandle caching ovis for Val.moreVars.
     */
    private static MethodHandle cacheMvMH(
            int symHandle,
            OwnVarIndexes ovis,
            int varInd) throws Throwable {
        var mt = MethodType.methodType(
                    void.class,
                    int.class,
                    Val.class,
                    OwnVarIndexes.class,
                    int.class);
        MethodHandle method = MethodHandles.lookup().findVirtual(Val.class, "setPredictMv", mt);
        var boundSymhandle = MethodHandles.insertArguments(method, 1, symHandle);
        return MethodHandles.insertArguments(boundSymhandle, 2, ovis, varInd);
    }

}

// vim: et sw=4 sts=4 fdm=marker
