package org.kink_lang.kink;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * A str val.
 */
public class StrVal extends Val {

    /** The root node of the binary tree representing a string. */
    private Node node;

    /** The count of runes in the str. */
    private int runeCount = UNINITIALIZED_RUNE_COUNT;

    /** runeCount field has not been initialized. */
    private static final int UNINITIALIZED_RUNE_COUNT = -1;

    /** The cache of (upper 32bits = rune pos, lower 32bits = char pos). */
    private long posCache;

    /**
     * Constructs a str val.
     *
     * @param vm the vm.
     * @param string the string.
     */
    StrVal(Vm vm, String string) {
        this(vm, new Leaf(string));
    }

    /**
     * Constructs a str val with the binary tree.
     */
    private StrVal(Vm vm, Node node) {
        super(vm, null);
        Preconds.checkArg(node.length() <= vm.str.getMaxLength(),
                "length must not exceed the max");
        this.node = node;
    }

    /**
     * Returns the string.
     *
     * @return the string.
     */
    public String string() {
        Leaf leaf = this.node.asLeaf();
        this.node = leaf;
        return leaf.string();
    }

    /**
     * Returns the count of runes in the str.
     */
    int runeCount() {
        if (this.runeCount == UNINITIALIZED_RUNE_COUNT) {
            var s = string();
            this.runeCount = s.codePointCount(0, s.length());
        }
        return this.runeCount;
    }

    /**
     * Returns the char length of the string.
     */
    int length() {
        return this.node.length();
    }

    /**
     * Converts the rune pos index to the corresponding char pos index.
     */
    int runePosToCharPos(int runePos) {
        long cache = this.posCache;
        int cachedRunePos = (int) (cache >>> 32);
        int cachedCharPos = (int) cache;
        int charPos = string().offsetByCodePoints(cachedCharPos, runePos - cachedRunePos);
        this.posCache = (((long) runePos) << 32) | Integer.toUnsignedLong(charPos);
        return charPos;
    }

    /**
     * Returns a str val concatenating {@code this} and {@code tail}.
     *
     * @param tail str val to concatenate.
     * @return a str val concatenating {@code this} and {@code tail}.
     */
    public StrVal concat(StrVal tail) {
        Preconds.checkArg(
                (long) length() + tail.length() <= vm.str.getMaxLength(),
                "length must not exceed the max");

        if (length() == 0) {
            return tail;
        } else if (tail.length() == 0) {
            return this;
        }

        Node left = this.node;
        Node right = tail.node;
        int leafCountSum = left.leafCount() + right.leafCount();
        Node resultNode = leafCountSum < 512
            ? new Tree(left, right)
            : Leaf.substantiate(List.of(right, left));
        return new StrVal(vm, resultNode);
    }

    @Override
    SharedVars sharedVars() {
        return vm.str.sharedVars;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "StrVal(%s)", string());
    }

    /**
     * Returns properties which determine equality of the type.
     */
    private List<Object> properties() {
        return List.of(this.vm, string());
    }

    @Override
    public int hashCode() {
        return properties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof StrVal argStr
            && properties().equals(argStr.properties());
    }

    /**
     * A binary tree node containing a part of string.
     */
    private static abstract class Node {

        /**
         * Returns the char length of the string.
         */
        abstract int length();

        /**
         * Returns the count of leaves under the node.
         */
        abstract int leafCount();

        /**
         * Returns a leaf equivalent to this node.
         */
        abstract Leaf asLeaf();

    }

    /**
     * A node containing two branches.
     */
    private static class Tree extends Node {

        /** The length of the string. */
        private final int length;

        /** The count of the leaves under the node. */
        private final int leafCount;

        /** The left branch. */
        private final Node left;

        /** The right branch. */
        private final Node right;

        /**
         * Constructs a node.
         */
        Tree(Node left, Node right) {
            this.length = left.length() + right.length();
            this.leafCount = left.leafCount() + right.leafCount();
            this.left = left;
            this.right = right;
        }

        /**
         * Returns the left branch.
         */
        Node left() {
            return this.left;
        }

        /**
         * Returns the right branch.
         */
        Node right() {
            return this.right;
        }

        @Override
        int length() {
            return this.length;
        }

        @Override
        int leafCount() {
            return this.leafCount;
        }

        @Override
        Leaf asLeaf() {
            return Leaf.substantiate(List.of(this));
        }

    }

    /**
     * A node containing a single string.
     */
    private static class Leaf extends Node {

        /** The string. */
        private final String string;

        /**
         * Constructs a node.
         */
        Leaf(String string) {
            this.string = string;
        }

        @Override
        int length() {
            return this.string.length();
        }

        @Override
        int leafCount() {
            return 1;
        }

        @Override
        Leaf asLeaf() {
            return this;
        }

        /**
         * Returns the content string.
         */
        String string() {
            return this.string;
        }

        /**
         * Makes a leaf node from the initial state of the work stack.
         */
        static Leaf substantiate(List<Node> initialStack) {
            int length = initialStack.stream().mapToInt(n -> n.length()).sum();
            List<Node> stack = new ArrayList<>(initialStack);
            StringBuilder builder = new StringBuilder(length);
            while (! stack.isEmpty()) {
                Node node = stack.remove(stack.size() - 1);
                if (node instanceof Leaf leaf) {
                    builder.append(leaf.string());
                } else {
                    Tree tree = (Tree) node;
                    stack.add(tree.right());
                    stack.add(tree.left());
                }
            }
            return new Leaf(builder.toString());
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
