package org.kink_lang.kink;

import java.lang.invoke.MethodHandles;
import java.lang.invoke.VarHandle;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;

import javax.annotation.Nullable;

import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.internal.control.Control;
import org.kink_lang.kink.internal.ovis.OwnVarIndexes;

/**
 * A Kink <a href="../../../../../manual/language/object.html#values">value</a> or val.
 */
public class Val extends HostResultCore implements HostResult {

    /** The vm. */
    public final Vm vm;

    /** The shared vars. If the field is null, sharedVars() must be overridden. */
    @Nullable
    private final SharedVars sharedVars;

    /** Mapping from syms to indexes of own vars. */
    private OwnVarIndexes ovis;

    /** Accessor of this.ovis. */
    private static final VarHandle OVIS_VH = Control.runWrappingThrowable(
            () -> MethodHandles.lookup()
            .findVarHandle(Val.class, "ovis", OwnVarIndexes.class));

    /** Own vars 0..5. */
    private Val v0, v1, v2, v3, v4, v5;

    /** Own vars 6...  */
    private Val[] moreVars;

    /** Empty array of values. */
    static final Val[] EMPTY_VS = new Val[0];

    /** Accessor of this.moreVars. */
    private static final VarHandle MOREVARS_VH = Control.runWrappingThrowable(
            () -> MethodHandles.lookup()
            .findVarHandle(Val.class, "moreVars", Val[].class));

    /**
     * Constructs a val.
     *
     * @param vm the vm.
     */
    protected Val(Vm vm) {
        this(vm, vm.sharedVars.minimal,
                vm.emptyOvis, null, null, null, null, null, null, EMPTY_VS);
    }

    /**
     * Constructs a val with the shared vars.
     *
     * @param vm the vm.
     * @param sharedVars the shared vars.
     */
    protected Val(Vm vm, SharedVars sharedVars) {
        // Internal child classes such as StrVal can pass null for sharedVars.
        // They must override sharedVars() instead.
        this(vm, sharedVars,
                vm.emptyOvis, null, null, null, null, null, null, EMPTY_VS);
    }

    /**
     * The only non-delegating constructor.
     */
    Val(Vm vm, SharedVars sharedVars,
            OwnVarIndexes ovis, Val v0, Val v1, Val v2, Val v3, Val v4, Val v5, Val[] moreVars) {
        this.vm = vm;
        this.sharedVars = sharedVars;
        this.ovis = ovis;
        this.v0 = v0;
        this.v1 = v1;
        this.v2 = v2;
        this.v3 = v3;
        this.v4 = v4;
        this.v5 = v5;
        this.moreVars = moreVars;
    }

    /**
     * Copying constructor.
     */
    Val(Val orig) {
        this(orig.vm, orig.sharedVars,
                orig.ovis, orig.v0, orig.v1, orig.v2, orig.v3, orig.v4, orig.v5,
                copyOf(orig.moreVars));
    }

    /**
     * Copy the array of values.
     */
    private static Val[] copyOf(Val[] moreVars) {
        if (moreVars.length == 0) {
            return moreVars;
        }
        Val[] newMvs = new Val[moreVars.length];
        System.arraycopy(moreVars, 0, newMvs, 0, moreVars.length);
        return newMvs;
    }

    @Override
    final void doTransition(StackMachine stackMachine) {
        stackMachine.getDataStack().removeFromOffset(0);
        stackMachine.transitionToResult(this);
    }

    @Override
    public final HostResultCore makeHostResultCore() {
        return this;
    }

    // identity {{{1

    /** Counter of unique numbers. */
    private static final AtomicReference<BigDecimal> UNIQUE_PRODUCER
        = new AtomicReference<>(BigDecimal.ZERO);

    /** Updater of {@code identity}. */
    private static final AtomicReferenceFieldUpdater<Val, BigDecimal> IDENTITY_UPDATER
        = AtomicReferenceFieldUpdater.newUpdater(Val.class, BigDecimal.class, "identity");

    /** Identity of the val, set only once on demand. */
    private volatile BigDecimal identity;

    /**
     * Returns the identity integral number of this val.
     *
     * The result can be converted into BigInteger exactly.
     * The reason why storing BigDecimal instead of BigInteger is
     * that NumVal consists of a BigDecimal number,
     * thus storing BigDecimal can improve computational efficiency a bit.
     */
    final BigDecimal identityDecimal() {
        if (this.identity == null) {
            BigDecimal newNum = UNIQUE_PRODUCER.updateAndGet(n -> n.add(BigDecimal.ONE));
            IDENTITY_UPDATER.compareAndSet(this, null, newNum);
        }
        return this.identity;
    }

    /**
     * Returns the identity number of this val.
     *
     * <p>The identity number is unique within the vm.</p>
     *
     * @return the identity number of this val.
     */
    public final BigInteger identity() {
        return identityDecimal().toBigInteger();
    }

    // }}}1

    // vars {{{1



    /**
     * Assigns {@code val} to the var specified by {@code symHandle}.
     *
     * @param symHandle the sym handle of the var.
     * @param val the val to be assigned.
     */
    public final void setVar(int symHandle, Val val) {
        int index = addAndGetIndex(symHandle);
        setAtIndex(index, val);
    }

    /**
     * Returns the target of the var specified by {@code sym}; or null if absent.
     *
     * @param symHandle the sym handle of the var.
     * @return the target of the var specified by {@code sym}; or null if absent.
     */
    public final @Nullable Val getVar(int symHandle) {
        Val ownVar = getOwnVar(symHandle);
        return ownVar != null ? ownVar : getShared(symHandle);
    }

    /**
     * Returns whether the val has the var specified by {@code sym}.
     *
     * @param symHandle the sym handle of the varl.
     * @return true if the val has the var specified by {@code sym}.
     */
    public final boolean hasVar(int symHandle) {
        return getVar(symHandle) != null;
    }

    /**
     * Returns the sym handle set of the vars of this val.
     *
     * @return the sym handle set of the vars of this val.
     */
    public final Set<Integer> getVarSymHandleSet() {
        Set<Integer> symHandles = new HashSet<>(sharedVars().symHandleSet());
        symHandles.addAll(getOwnVarSymHandleSet());
        return Collections.unmodifiableSet(symHandles);
    }

    /**
     * Returns the shared vars for the val.
     *
     * <p>This method can be overridden in order to allow lazy initialization of traits.</p>
     *
     * @return the shared vars for the val.
     */
    SharedVars sharedVars() {
        return this.sharedVars;
    }

    /**
     * Returns the shared var.
     */
    private Val getShared(int symHandle) {
        return sharedVars().get(symHandle);
    }

    // }}}1

    // own-vars {{{1

    /**
     * Ensure symHandle is set in this.ovis, and returns its index.
     */
    final int addAndGetIndex(int symHandle) {
        while (true) {
            OwnVarIndexes origOvis = this.ovis;
            int index = origOvis.getIndex(symHandle);
            if (index >= 0) {
                return index;
            }

            OwnVarIndexes newOvis = origOvis.with(symHandle);
            OVIS_VH.compareAndSet(this, origOvis, newOvis);
        }
    }

    /**
     * Sets the own var indexes.
     *
     * Precondition: this.ovis must be empty.
     */
    final void setOvis(OwnVarIndexes ovis) {
        OVIS_VH.set(this, ovis);
    }

    /**
     * Gets the own var indexes.
     */
    final OwnVarIndexes getOvis() {
        return this.ovis;
    }

    /**
     * Whether the val does not have own var.
     */
    final boolean noOwnVar() {
        return this.ovis.isEmpty();
    }

    /**
     * Returns the specified own-var, or null if absent.
     */
    @Nullable final Val getOwnVar(int symHandle) {
        int index = this.ovis.getIndex(symHandle);
        return index >= 0 ? getOwnAtIndex(index) : null;
    }

    /**
     * Returns the set of the sym handles of the own-vars.
     */
    final Set<Integer> getOwnVarSymHandleSet() {
        List<Integer> symHandles = this.ovis.getSymHandles();
        return IntStream.range(0, symHandles.size())
            .filter(i -> getOwnAtIndex(i) != null)
            .mapToObj(symHandles::get)
            .collect(Collectors.toUnmodifiableSet());
    }

    /**
     * Gets an own var at the index.
     */
    @Nullable
    final Val getOwnAtIndex(int index) {
        return switch (index) {
            case 0 -> this.v0;
            case 1 -> this.v1;
            case 2 -> this.v2;
            case 3 -> this.v3;
            case 4 -> this.v4;
            case 5 -> this.v5;
            default -> getMv(index);
        };
    }

    /**
     * Gets the var from moreVars.
     */
    @Nullable
    private Val getMv(int varInd) {
        int mvsInd = toMvsInd(varInd);
        Val[] moreVars = this.moreVars;
        return mvsInd < moreVars.length ? moreVars[mvsInd] : null;
    }

    /**
     * Sets an own var at the index.
     */
    final void setAtIndex(int index, Val val) {
        switch (index) {
            case 0 -> setV0(val);
            case 1 -> setV1(val);
            case 2 -> setV2(val);
            case 3 -> setV3(val);
            case 4 -> setV4(val);
            case 5 -> setV5(val);
            default -> setMv(index, val);
        }
    }

    /**
     * Converts the index from OwnVarIndexes to the index of moreVars.
     */
    private static int toMvsInd(int index) {
        return index - 6;
    }

    /**
     * Sets {@code val} to v0.
     */
    final void setV0(Val val) {
        this.v0 = val;
    }

    /**
     * Sets {@code val} to v1.
     */
    final void setV1(Val val) {
        this.v1 = val;
    }

    /**
     * Sets {@code val} to v2.
     */
    final void setV2(Val val) {
        this.v2 = val;
    }

    /**
     * Sets {@code val} to v3.
     */
    final void setV3(Val val) {
        this.v3 = val;
    }

    /**
     * Sets {@code val} to v4.
     */
    final void setV4(Val val) {
        this.v4 = val;
    }

    /**
     * Sets {@code val} to v5.
     */
    final void setV5(Val val) {
        this.v5 = val;
    }

    /**
     * Sets {@code val} to an element of moreVars.
     */
    final void setMv(int varInd, Val val) {
        int mvsInd = toMvsInd(varInd);
        while (true) {
            Val[] origMvs = (Val[]) MOREVARS_VH.getAcquire(this);
            Val[] newMvs = makeMoreVars(origMvs, mvsInd, val);
            if (MOREVARS_VH.weakCompareAndSet(this, origMvs, newMvs)) {
                break;
            }
        }
    }

    /**
     * Make new moreVars. Overridable for a test.
     */
    Val[] makeMoreVars(Val[] origMvs, int mvsInd, Val val) {
        Val[] newMvs = new Val[Math.max(origMvs.length, mvsInd + 1)];
        System.arraycopy(origMvs, 0, newMvs, 0, origMvs.length);
        newMvs[mvsInd] = val;
        return newMvs;
    }

    // }}}1

    // get with prediction {{{

    // The purpose of the following methods is to avoid symHandle searching in OwnVarIndexes,
    // and SharedVars.getIndex.

    /**
     * Returns the var at symHandle or null,
     * with prediction that this.ovis == predictedOvis.
     *
     * Precondition: the #0 sym handle of {@code predictedOvis} arg must be {@code symHandle}.
     */
    @Nullable
    final Val getPredictV0(int symHandle, OwnVarIndexes predictedOvis) {
        if (this.ovis != predictedOvis) {
            return getVar(symHandle);
        }

        Val own = this.v0;
        return own != null ? own : getShared(symHandle);
    }

    /**
     * Returns the var at symHandle or null,
     * with prediction that this.ovis == predictedOvis.
     *
     * Precondition: the #1 sym handle of {@code predictedOvis} arg must be {@code symHandle}.
     */
    @Nullable
    final Val getPredictV1(int symHandle, OwnVarIndexes predictedOvis) {
        if (this.ovis != predictedOvis) {
            return getVar(symHandle);
        }

        Val own = this.v1;
        return own != null ? own : getShared(symHandle);
    }

    /**
     * Returns the var at symHandle or null,
     * with prediction that this.ovis == predictedOvis.
     *
     * Precondition: the #2 sym handle of {@code predictedOvis} arg must be {@code symHandle}.
     */
    @Nullable
    final Val getPredictV2(int symHandle, OwnVarIndexes predictedOvis) {
        if (this.ovis != predictedOvis) {
            return getVar(symHandle);
        }

        Val own = this.v2;
        return own != null ? own : getShared(symHandle);
    }

    /**
     * Returns the var at symHandle or null,
     * with prediction that this.ovis == predictedOvis.
     *
     * Precondition: the #3 sym handle of {@code predictedOvis} arg must be {@code symHandle}.
     */
    @Nullable
    final Val getPredictV3(int symHandle, OwnVarIndexes predictedOvis) {
        if (this.ovis != predictedOvis) {
            return getVar(symHandle);
        }

        Val own = this.v3;
        return own != null ? own : getShared(symHandle);
    }

    /**
     * Returns the var at symHandle or null,
     * with prediction that this.ovis == predictedOvis.
     *
     * Precondition: the #4 sym handle of {@code predictedOvis} arg must be {@code symHandle}.
     */
    @Nullable
    final Val getPredictV4(int symHandle, OwnVarIndexes predictedOvis) {
        if (this.ovis != predictedOvis) {
            return getVar(symHandle);
        }

        Val own = this.v4;
        return own != null ? own : getShared(symHandle);
    }

    /**
     * Returns the var at symHandle or null,
     * with prediction that this.ovis == predictedOvis.
     *
     * Precondition: the #5 sym handle of {@code predictedOvis} arg must be {@code symHandle}.
     */
    @Nullable
    final Val getPredictV5(int symHandle, OwnVarIndexes predictedOvis) {
        if (this.ovis != predictedOvis) {
            return getVar(symHandle);
        }

        Val own = this.v5;
        return own != null ? own : getShared(symHandle);
    }

    /**
     * Returns the var at symHandle or null,
     * with prediction that this.ovis == predictedOvis.
     *
     * Precondition: the #varInd sym handle of {@code predictedOvis}
     * arg must be {@code symHandle}.
     */
    @Nullable
    final Val getPredictMv(int symHandle, OwnVarIndexes predictedOvis, int varInd) {
        if (this.ovis != predictedOvis) {
            return getVar(symHandle);
        }

        Val own = getMv(varInd);
        return own != null ? own : getShared(symHandle);
    }

    /**
     * Returns the var at symHandle or null,
     * with prediction that this.ovis == predictedOvis
     * and this.sharedVars().uniqueId == svUniqueId.
     *
     * Preconditions:
     *
     * • predictedOvis should not contain symHandle.
     *
     * • The sym handle of #svInd of the SharedVars with svUniqueId should be symHandle.
     */
    @Nullable
    final Val getPredictShared(
            int symHandle, OwnVarIndexes predictedOvis, long svUniqueId, int svInd) {
        if (this.ovis != predictedOvis) {
            return getVar(symHandle);
        }

        SharedVars sv = sharedVars();
        return sv.getUniqueId() == svUniqueId
            ? sv.getAtIndex(svInd)
            : sv.get(symHandle);
    }

    // }}}

    // set with prediction {{{

    // The purpose of the following methods is to avoid symHandle searching in OwnVarIndexes.

    /**
     * Sets {@code val} at {@code symHandle} with prediction that this.ovis == predictedOvis.
     *
     * Precondition:
     *
     * • The #0 sym handle of {@code predictedOvis} must be {@code symHandle}.
     */
    final void setPredictV0(int symHandle, Val val, OwnVarIndexes predictedOvis) {
        if (this.ovis == predictedOvis) {
            setV0(val);
        } else {
            setVar(symHandle, val);
        }
    }

    /**
     * Sets {@code val} at {@code symHandle} with prediction that this.ovis == predictedOvis.
     *
     * Precondition:
     *
     * • The #1 sym handle of {@code predictedOvis} must be {@code symHandle}.
     */
    final void setPredictV1(int symHandle, Val val, OwnVarIndexes predictedOvis) {
        if (this.ovis == predictedOvis) {
            setV1(val);
        } else {
            setVar(symHandle, val);
        }
    }

    /**
     * Sets {@code val} at {@code symHandle} with prediction that this.ovis == predictedOvis.
     *
     * Precondition:
     *
     * • The #2 sym handle of {@code predictedOvis} must be {@code symHandle}.
     */
    final void setPredictV2(int symHandle, Val val, OwnVarIndexes predictedOvis) {
        if (this.ovis == predictedOvis) {
            setV2(val);
        } else {
            setVar(symHandle, val);
        }
    }

    /**
     * Sets {@code val} at {@code symHandle} with prediction that this.ovis == predictedOvis.
     *
     * Precondition:
     *
     * • The #3 sym handle of {@code predictedOvis} must be {@code symHandle}.
     */
    final void setPredictV3(int symHandle, Val val, OwnVarIndexes predictedOvis) {
        if (this.ovis == predictedOvis) {
            setV3(val);
        } else {
            setVar(symHandle, val);
        }
    }

    /**
     * Sets {@code val} at {@code symHandle} with prediction that this.ovis == predictedOvis.
     *
     * Precondition:
     *
     * • The #4 sym handle of {@code predictedOvis} must be {@code symHandle}.
     */
    final void setPredictV4(int symHandle, Val val, OwnVarIndexes predictedOvis) {
        if (this.ovis == predictedOvis) {
            setV4(val);
        } else {
            setVar(symHandle, val);
        }
    }

    /**
     * Sets {@code val} at {@code symHandle} with prediction that this.ovis == predictedOvis.
     *
     * Precondition:
     *
     * • The #5 sym handle of {@code predictedOvis} must be {@code symHandle}.
     */
    final void setPredictV5(int symHandle, Val val, OwnVarIndexes predictedOvis) {
        if (this.ovis == predictedOvis) {
            setV5(val);
        } else {
            setVar(symHandle, val);
        }
    }

    /**
     * Sets {@code val} at {@code symHandle} with prediction that this.ovis == predictedOvis.
     *
     * Precondition:
     *
     * • The #varInd sym handle of {@code predictedOvis} must be {@code symHandle}.
     */
    final void setPredictMv(int symHandle, Val val, OwnVarIndexes predictedOvis, int varInd) {
        if (this.ovis == predictedOvis) {
            setMv(varInd, val);
        } else {
            setVar(symHandle, val);
        }
    }

    // }}}

}

// vim: et sw=4 sts=4 fdm=marker
