package org.kink_lang.kink.internal.callstack;

import java.util.List;
import java.util.stream.LongStream;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Slice of call stack.
 *
 * @param callStackElements call stack elements of the slice.
 * @param tailTraceRingBuffers  trace ring buffers of the slice.
 * @param lnums See {@link Lnums}.
 */
public record CallStackSlice(
        Cse[] callStackElements,
        TailTraceRingBuffer[] tailTraceRingBuffers,
        long[] lnums) {

    /**
     * Constructs a slice.
     *
     * @param callStackElements call stack elements of the slice.
     * @param tailTraceRingBuffers  trace ring buffers of the slice.
     * @param lnums See {@link Lnums}.
     */
    public CallStackSlice {
        Preconds.checkArg(callStackElements.length == tailTraceRingBuffers.length,
                "#callStackElements must be equal to #tailTraceRingBuffers");
        Preconds.checkArg(callStackElements.length == lnums.length,
                "#callStackElements must be equal to #lnums");
    }

    /**
     * Returns the size of the slice.
     *
     * @return the size of the slice.
     */
    public int size() {
        return callStackElements().length;
    }

    /**
     * Returns the total usage of the dataStack by the slice.
     *
     * @return the total usage of the dataStack by the slice.
     */
    public int dataStackUsage() {
        int sum = 0;
        for (long lnum : this.lnums) {
            sum += Lnums.getDataStackUsage(lnum);
        }
        return sum;
    }

    /**
     * Returns the equality properties of the slice.
     */
    private List<Object> properties() {
        return List.of(
                List.of(this.callStackElements),
                List.of(this.tailTraceRingBuffers),
                LongStream.of(lnums()).mapToObj(n -> n).toList());
    }

    @Override
    public int hashCode() {
        return properties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
                || arg instanceof CallStackSlice argSlice
                && properties().equals(argSlice.properties());
    }

}

// vim: et sw=4 sts=4 fdm=marker
