package org.kink_lang.kink.internal.compile.javaclassir;

import java.util.ArrayList;
import java.util.List;
import java.util.function.BiFunction;

import org.objectweb.asm.Type;
import org.objectweb.asm.commons.Method;

import org.kink_lang.kink.FunVal;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.program.itree.*;

/**
 * Generates letrec within fast fun.
 */
public class InFastFunLetRecGenerator implements LetRecGenerator {

    /** Lvar access generator. */
    private final LvarAccessGenerator lvarAccGen;

    /** Make fun generator. */
    private final MakeFastFunGenerator makeFunGen;

    /** Type of DelegatingFunVal. */
    static final Type DELEGATING_FUN_TYPE = Type.getType("Lorg/kink_lang/kink/DelegatingFunVal;");

    /** Insn of DelegatingFunVal.wrap. */
    static final Insn INVOKE_WRAP = new Insn.InvokeVirtual(
            DELEGATING_FUN_TYPE,
            new Method("wrap", Type.VOID_TYPE, new Type[] { Type.getType(FunVal.class) }));

    /**
     * Constructs generator.
     *
     * @param lvarAccGen the lvar access generator.
     * @param makeFunGen the make fun generator.
     */
    public InFastFunLetRecGenerator(
            LvarAccessGenerator lvarAccGen,
            MakeFastFunGenerator makeFunGen) {
        this.lvarAccGen = lvarAccGen;
        this.makeFunGen = makeFunGen;
    }

    @Override
    public List<Insn> letRec(
            LetRecItree itree,
            BiFunction<Itree, ResultContext, List<Insn>> generate) {
        List<Insn> insns = new ArrayList<>();
        for (LetRecItree.LvarFunPair pair : itree.lvarFunPairs()) {
            LocalVar lvar = pair.lvar();
            if (! lvarAccGen.isUnused(lvar)) {
                // contParam = new DelegatingFunVal(vm)
                insns.add(new Insn.NewInstance(DELEGATING_FUN_TYPE));
                insns.add(new Insn.Dup());
                insns.addAll(InsnsGenerator.LOAD_VM);
                insns.add(new Insn.InvokeConstructor(DELEGATING_FUN_TYPE,
                            new Method("<init>", Type.VOID_TYPE,
                                new Type[] { Type.getType(Vm.class) })));
                insns.add(InsnsGenerator.STORE_CONTPARAM);

                // store contParam to lvar
                insns.addAll(lvarAccGen.storeLvar(lvar));
            }
        }

        for (LetRecItree.LvarFunPair pair : itree.lvarFunPairs()) {
            LocalVar lvar = pair.lvar();
            if (! lvarAccGen.isUnused(lvar)) {
                // contParam = fun
                insns.addAll(lvarAccGen.loadLvarAllowNull(lvar));

                // ((DelegatingFunVal) contParam).wrap(fun)
                insns.add(InsnsGenerator.LOAD_CONTPARAM);
                insns.add(new Insn.CheckCast(DELEGATING_FUN_TYPE));
                insns.addAll(makeFunGen.makeFun(pair.fun()));
                insns.add(INVOKE_WRAP);
            }
        }

        return insns;
    }

}

// vim: et sw=4 sts=4 fdm=marker
