package org.kink_lang.kink.internal.compile.javaclassir;

import java.lang.invoke.MethodType;
import java.util.ArrayList;
import java.util.List;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.callstack.Location;
import org.kink_lang.kink.internal.program.itree.LocalVar;

/**
 * Generates access to local vars of slow funs.
 */
public class SlowLvarAccessGenerator implements LvarAccessGenerator {

    /** The vm. */
    private final Vm vm;

    /** Generates binding. */
    private final BindingGenerator bindingGen;

    /** The key str supplier. */
    private final KeyStrSupplier keySup;

    /** The trace accumulator. */
    private final TraceAccumulator traceAccum;

    /**
     * Constructs a generator.
     *
     * @param vm the vm.
     * @param bindingGen generates binding.
     * @param keySup the key str supplier.
     * @param traceAccum the trace accumulator.
     */
    public SlowLvarAccessGenerator(
            Vm vm,
            BindingGenerator bindingGen,
            KeyStrSupplier keySup,
            TraceAccumulator traceAccum) {
        this.vm = vm;
        this.bindingGen = bindingGen;
        this.keySup = keySup;
        this.traceAccum = traceAccum;
    }

    @Override
    public List<Insn> loadLvar(LocalVar lvar, Location loc) {
        List<Insn> insns = new ArrayList<>(loadLvarAllowNull(lvar));
        insns.addAll(LvarAccessGenerator.checkNull(lvar.name(), loc, keySup, traceAccum));
        return insns;
    }

    @Override
    public List<Insn> loadLvarAllowNull(LocalVar lvar) {
        // contParam = get-var[sym](binding)
        List<Insn> insns = new ArrayList<>(bindingGen.generateBinding());
        insns.add(new Insn.InvokeDynamic(
                    MethodType.methodType(Val.class, Val.class),
                    InsnsGenerator.BOOTSTRAP_GET_VAR_HANDLE,
                    List.of(vm.sym.handleFor(lvar.name()))));
        insns.add(InsnsGenerator.STORE_CONTPARAM);
        return insns;
    }

    @Override
    public List<Insn> storeLvar(LocalVar lvar) {
        return setVar(lvar, List.of(InsnsGenerator.LOAD_CONTPARAM));
    }

    @Override
    public List<Insn> passRecv(LocalVar lvar) {
        return setVar(lvar, List.of(
                    InsnsGenerator.LOAD_DATASTACK,
                    InsnsGenerator.INVOKE_RECV));
    }

    @Override
    public List<Insn> passArg(LocalVar lvar, int argIndex) {
        return setVar(lvar, List.of(
                    InsnsGenerator.LOAD_DATASTACK,
                    new Insn.PushInt(argIndex),
                    InsnsGenerator.INVOKE_ARG));
    }

    /**
     * Generates insns to set a var of the binding.
     */
    private List<Insn> setVar(LocalVar lvar, List<Insn> content) {
        // set-var[sym](binding, content)
        List<Insn> insns = new ArrayList<>(bindingGen.generateBinding());
        insns.addAll(content);
        insns.add(new Insn.InvokeDynamic(
                    MethodType.methodType(void.class, Val.class, Val.class),
                    InsnsGenerator.BOOTSTRAP_SET_VAR_HANDLE,
                    List.of(vm.sym.handleFor(lvar.name()))));
        return insns;
    }

    @Override
    public boolean isUnused(LocalVar lvar) {
        return false;
    }

}

// vim: et sw=4 sts=4 fdm=marker
