package org.kink_lang.kink.internal.mod.java;

import javax.annotation.Nullable;

import org.kink_lang.kink.JavaVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * Factory of JAVA_THROW mod.
 */
public class JavaThrowMod {

    /** The vm. */
    private final Vm vm;

    /** The name of the mod. */
    public static final String MOD_NAME = "kink/javahost/JAVA_THROW";

    /**
     * Constructs the factory.
     */
    private JavaThrowMod(Vm vm) {
        this.vm = vm;
    }

    /**
     * Makes JAVA_THROW mod in the vm.
     *
     * @param vm the vm.
     * @return JAVA_THROW mod.
     */
    public static Val makeMod(Vm vm) {
        return new JavaThrowMod(vm).makeMod();
    }

    /**
     * Maks a mod.
     */
    private Val makeMod() {
        Val mod = vm.newVal();
        mod.setVar(vm.sym.handleFor("new"),
                vm.fun.make("JAVA_THROW.new(Exception_java_val)")
                .take(1).action(this::newFun));
        mod.setVar(vm.sym.handleFor("is?"),
                vm.fun.make("JAVA_THROW.is?(Val)").take(1).action(this::isPFun));
        return mod;
    }

    /**
     * Implementation of JAVA_THROW.new.
     */
    private HostResult newFun(CallContext c) {
        Throwable th = extractThrowable(c.arg(0));
        if (th == null) {
            String desc = "JAVA_THROW.new(Exception_java_val)";
            return c.call(vm.graph.raiseFormat(
                        "{}: the arg must be a java val of a Throwable, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        return JavaThrowVal.of(vm, th);
    }

    /**
     * Extracts a Throwable instance from arg,
     * or null if it is not an appropriate java val.
     */
    @Nullable
    private Throwable extractThrowable(Val arg) {
        if (! (arg instanceof JavaVal excJavaVal)) {
            return null;
        }
        if (! (excJavaVal.objectReference() instanceof Throwable th)) {
            return null;
        }
        return th;
    }

    /**
     * Implementation of JAVA_THROW.is?.
     */
    private HostResult isPFun(CallContext c) {
        return vm.bool.of(c.arg(0) instanceof JavaThrowVal);
    }

}

// vim: et sw=4 sts=4 fdm=marker
