package org.kink_lang.kink.internal.program.itree;

import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

/**
 * A fun in SSA (static single assignment) form.
 */
public class FastFunItree implements Itree, UsedDefinedVars {

    /** The fun body. */
    private final Itree body;

    /** The pos of the itree on the program text. */
    private final int pos;

    /** Lvars used in the fun body. */
    private final Set<LocalVar> usedLvars;

    /** Lvars defined in the fun body. */
    private final Set<LocalVar> definedLvars;

    /** Free lvars of the fun. */
    private final Set<LocalVar> freeLvars;

    /** Mapping from local vars to the contents. */
    private final Map<LocalVar, LocalVarContent> lvarContentMapping;

    /**
     * Constructs a fun itree in SSA form.
     *
     * @param body the fun body itree; must not expose the env.
     * @param pos the start pos of the fun.
     */
    public FastFunItree(Itree body, int pos) {
        this.body = body;
        this.pos = pos;

        var analysis = UsageDefinitionAnalysis.analyzeUseDefine(this.body);
        this.usedLvars = analysis.usedLvars();
        this.definedLvars = analysis.definedLvars();
        this.freeLvars = analysis.freeLvars();

        this.lvarContentMapping = ContentAnalysis.analyzeContent(this.body);
    }

    /**
     * Returns the fun body itree.
     *
     * @return the fun body itree.
     */
    public Itree body() {
        return this.body;
    }

    @Override
    public int pos() {
        return this.pos;
    }

    @Override
    public Set<LocalVar> usedLvars() {
        return this.usedLvars;
    }

    @Override
    public Set<LocalVar> definedLvars() {
        return this.definedLvars;
    }

    @Override
    public Set<LocalVar> freeLvars() {
        return this.freeLvars;
    }

    /**
     * Mapping from the local var to the content.
     *
     * @return the mapping.
     */
    public Map<LocalVar, LocalVarContent> lvarContentMapping() {
        return this.lvarContentMapping;
    }

    @Override
    public LocalVarContent getContent(LocalVar lvar) {
        return lvarContentMapping().getOrDefault(lvar, new LocalVarContent.Unknown());
    }

    @Override
    public <T> T accept(ItreeVisitor<T> visitor) {
        return visitor.visit(this);
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "FastFunItree(%s %d)", this.body, this.pos);
    }

    /**
     * Returns the properties of the fun itree.
     */
    private List<Object> getProperties() {
        return List.of(this.body, this.pos);
    }

    @Override
    public int hashCode() {
        return getProperties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof FastFunItree fun
            && getProperties().equals(fun.getProperties());
    }

}

// vim: et sw=4 sts=4 fdm=marker
