package org.kink_lang.kink.internal.program.itree;

/**
 * A visitor of {@link Itree}s.
 *
 * @param <T> the result type of visitor methods.
 */
public interface ItreeVisitor<T> {

    /**
     * Visits a context binding itree.
     *
     * @param binding the context binding itree.
     * @return the result.
     */
    public abstract T visit(BindingItree binding);

    /**
     * Visits a context recv itree.
     *
     * @param recv the context recv itree.
     * @return the result.
     */
    public abstract T visit(RecvItree recv);

    /**
     * Visits a context arg vec itree.
     *
     * @param argVec the context arg vec itree.
     * @return the result.
     */
    public abstract T visit(ArgVecItree argVec);

    /**
     * Visits a num itree.
     *
     * @param num the num itree.
     * @return the result.
     */
    public abstract T visit(NumItree num);

    /**
     * Visits a str itree.
     *
     * @param str the str itree.
     * @return the result.
     */
    public abstract T visit(StrItree str);

    /**
     * Visits a nada itree.
     *
     * @param nada the nada itree.
     * @return the result.
     */
    public abstract T visit(NadaItree nada);

    /**
     * Visits a seq itree.
     *
     * @param seq the seq itree.
     * @return the result.
     */
    public abstract T visit(SeqItree seq);

    /**
     * Visits a fun itree.
     *
     * @param fun the fun itree.
     * @return the result.
     */
    public abstract T visit(SlowFunItree fun);

    /**
     * Visits an indexable fun itree.
     *
     * @param fun the indexable fun itree.
     * @return the result.
     */
    public abstract T visit(FastFunItree fun);

    /**
     * Visits a vec itree.
     *
     * @param vec the vec itree.
     * @return the result.
     */
    public abstract T visit(VecItree vec);

    /**
     * Visits a deref itree.
     *
     * @param deref the deref itree.
     * @return the result.
     */
    public abstract T visit(DerefItree deref);

    /**
     * Visits a lderef itree.
     *
     * @param lderef the lderef itree.
     * @return the result.
     */
    public abstract T visit(LderefItree lderef);

    /**
     * Visits a varref itree.
     *
     * @param varref the varref itree.
     * @return the result.
     */
    public abstract T visit(VarrefItree varref);

    /**
     * Visits an itree of letrec.
     *
     * @param itree the letrec itree.
     * @return the result.
     */
    public abstract T visit(LetRecItree itree);

    /**
     * Visits an assignment itree.
     *
     * @param assignment the assignment itree.
     * @return the result.
     */
    public abstract T visit(AssignmentItree assignment);

    /**
     * Visits an assignment itree in the form {@code [:A :B :C.opt :D.opt] <- xxx}.
     *
     * @param assignment the itree.
       @return the result.
     */
    public abstract T visit(OptVecAssignmentItree assignment);

    /**
     * Visits an assignment itree in the form {@code [:A :B :C.opt :D.opt :R.rest] <- xxx}.
     *
     * @param assignment the itree.
       @return the result.
     */
    public abstract T visit(OptRestVecAssignmentItree assignment);

    /**
     * Visits an assignment itree in the form {@code [:R.rest] <- xxx}.
     *
     * @param assignment the itree.
       @return the result.
     */
    public abstract T visit(RestVecAssignmentItree assignment);

    /**
     * Visits an assignment itree in the form {@code [:Foo [:X :Y]] <- xxx}.
     *
     * @param assignment the itree.
       @return the result.
     */
    public abstract T visit(NestedVecAssignmentItree assignment);

    /**
     * Visits an assignment itree in the form {@code [:Foo X:Bar] <- Rhs}.
     *
     * @param assignment the itree.
     * @return the result.
     */
    public abstract T visit(VarrefVecAssignmentItree assignment);

    /**
     * Visits a lstore itree.
     *
     * @param lstore the lstore itree.
     * @return the result.
     */
    public abstract T visit(LstoreItree lstore);

    /**
     * Visists a store itree.
     *
     * @param store the store itree.
     * @return the result.
     */
    public abstract T visit(StoreItree store);

    /**
     * Visists an args passing itree.
     *
     * @param itree the arg passing itree.
     * @return the result.
     */
    public abstract T visit(ArgsPassingItree itree);

    /**
     * Visists a nested args passing itree.
     *
     * @param itree the nested args passing itree.
     * @return the result.
     */
    public abstract T visit(NestedArgsPassingItree itree);

    /**
     * Visits an itree of {@code new_val(... Trait 'X' x 'Y' y)}.
     *
     * @param itree the new_val itree.
     * @return the result.
     */
    public abstract T visit(TraitNewValItree itree);

    /**
     * Visits a bi arithmetic operation.
     *
     * @param itree the arithmetic operation itree.
     * @return the result.
     */
    public abstract T visit(BiArithmeticItree itree);

    /**
     * Visits an itree of {@code new_val('X' x 'Y' y)}.
     *
     * @param itree the new_val itree.
     * @return the result.
     */
    public abstract T visit(NoTraitNewValItree itree);

    /**
     * Visits an itree of preloaded if on the tail context.
     *
     * @param itree the preloaded if itree.
     * @return the result.
     */
    public abstract T visit(IfItree itree);

    /**
     * Visits an itree of branch without an else block on the tail context.
     *
     * @param itree the branch itree.
     * @return the result.
     */
    public abstract T visit(BranchItree itree);

    /**
     * Visits an itree of branch with an else block on the tail context.
     *
     * @param itree the branch itree.
     * @return the result.
     */
    public abstract T visit(BranchWithElseItree itree);

    /**
     * Visits a mcall itree.
     *
     * @param mcall the mcall itree.
     * @return the result.
     */
    public abstract T visit(McallItree mcall);

    /**
     * Visits a symcall itree.
     *
     * @param scall the symcall itree.
     * @return the result.
     */
    public abstract T visit(SymcallItree scall);

}

// vim: et sw=4 sts=4 fdm=marker
