package org.kink_lang.kink.internal.program.itree;

import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.kink_lang.kink.internal.program.ast.*;

/**
 * A function which translates an AST node into an itree.
 */
public class NodeToItreeTranslator implements Function<Expr, Itree> {

    /** The visitor used for translation. */
    private final TranslateVisitor visitor = new TranslateVisitor();

    /**
     * Translates an AST node to an itree.
     *
     * @param expr the AST node to translate.
     * @return the itree translated from the AST node.
     */
    @Override
    public Itree apply(Expr expr) {
        return expr.accept(visitor);
    }

    /**
     * Translate an elem.
     */
    private ItreeElem translateElem(Elem elem) {
        if (elem instanceof  Expr) {
            return apply((Expr) elem);
        } else {
            Elem.Spread spread = (Elem.Spread) elem;
            return new ItreeElem.Spread(apply(spread.expr()), spread.pos());
        }
    }

    /**
     * Translate elems.
     */
    List<ItreeElem> translateElems(List<Elem> elems) {
        return elems.stream()
            .map(this::translateElem)
            .collect(Collectors.toList());
    }

    /**
     * A visitor used for translation.
     */
    private class TranslateVisitor implements ExprVisitor<Itree> {

        @Override
        public Itree visitSeq(SeqExpr seq) {
            if (seq.steps().isEmpty()) {
                return new NadaItree(seq.pos());
            } else {
                List<Itree> steps = seq.steps().stream()
                    .map(NodeToItreeTranslator.this::apply)
                    .collect(Collectors.toList());
                return new SeqItree(steps, seq.pos());
            }
        }

        @Override
        public Itree visitNum(NumExpr num) {
            return new NumItree(num.num(), num.pos());
        }

        @Override
        public Itree visitStr(StrExpr str) {
            return new StrItree(str.str(), str.pos());
        }

        @Override
        public Itree visitBinding(BindingExpr binding) {
            return new BindingItree(binding.pos());
        }

        @Override
        public Itree visitDeref(DerefExpr deref) {
            return new DerefItree(apply(deref.owner()), deref.sym(), deref.pos());
        }

        @Override
        public Itree visitVarref(VarrefExpr varref) {
            return new VarrefItree(apply(varref.owner()), varref.sym(), varref.pos());
        }

        @Override
        public Itree visitFun(FunExpr fun) {
            Itree body = apply(fun.body());
            int pos = fun.pos();
            Itree recvStore = new LstoreItree(
                    new LocalVar.Original("_Recv"), new RecvItree(pos), pos);
            Itree argsStore = new LstoreItree(
                    new LocalVar.Original("_Args"), new ArgVecItree(pos), pos);
            return new SlowFunItree(
                    new SeqItree(List.of(recvStore, argsStore, body), pos),
                    pos);
        }

        @Override
        public Itree visitVec(VecExpr vec) {
            List<ItreeElem> elems = translateElems(vec.elems());
            return new VecItree(elems, vec.pos());
        }

        @Override
        public Itree visitMcall(McallExpr mcall) {
            Itree ownerRecv = apply(mcall.ownerRecv());
            List<ItreeElem> args = translateElems(mcall.args());
            return new McallItree(ownerRecv, mcall.sym(), args, mcall.pos());
        }

        @Override
        public Itree visitRcall(RcallExpr rcall) {
            Itree owner = apply(rcall.owner());
            int pos = rcall.pos();
            String sym = rcall.sym();
            DerefItree fun = new DerefItree(owner, sym, pos);
            Itree recv = apply(rcall.recv());
            List<ItreeElem> args = translateElems(rcall.args());
            return new SymcallItree(fun, sym, recv, args, pos);
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
