package org.kink_lang.kink.internal.program.itreeoptimize;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.program.itree.AssignmentItree;
import org.kink_lang.kink.internal.program.itree.BindingItree;
import org.kink_lang.kink.internal.program.itree.Itree;
import org.kink_lang.kink.internal.program.itree.ItreeElem;
import org.kink_lang.kink.internal.program.itree.LocalVar;
import org.kink_lang.kink.internal.program.itree.NestedParam;
import org.kink_lang.kink.internal.program.itree.NestedVecAssignmentItree;
import org.kink_lang.kink.internal.program.itree.VarrefItree;
import org.kink_lang.kink.internal.program.itree.VecItree;

/**
 * Inlining of assignment in the form {@code [:X [:Y :Z]] <- Rhs}.
 */
public class NestedVecAssignmentInliner extends BaseOptimizer {

    @Override
    public Itree visit(AssignmentItree itree) {
        return extractParams(itree.lhs())
            .map(params -> (Itree) new NestedVecAssignmentItree(params, itree.rhs(), itree.pos()))
            .orElse(itree);
    }

    /**
     * Extracts the list of params when lhs is an vec of (lvar or (vec of lvars)).
     */
    private Optional<List<NestedParam>> extractParams(Itree lhs) {
        if (! (lhs instanceof VecItree vec)) {
            return Optional.empty();
        }
        List<ItreeElem> lhsElems = vec.elems();
        List<NestedParam> params = new ArrayList<>();
        for (ItreeElem elem : lhsElems) {
            @Nullable
            NestedParam param = extractParam(elem);
            if (param == null) {
                return Optional.empty();
            }
            params.add(param);
        }
        return Optional.of(params);
    }

    /**
     * Extracts a param from the elem, or returns null.
     */
    @Nullable
    private NestedParam extractParam(ItreeElem elem) {
        if (! (elem instanceof Itree lhs)) {
            return null;
        }

        if (lhs instanceof VarrefItree varref) {
            if (! (varref.owner() instanceof BindingItree)) {
                return null;
            }
            return new LocalVar.Original(varref.sym());
        } else if (lhs instanceof VecItree tuple) {
            List<LocalVar> lvars = new ArrayList<>();
            for (ItreeElem tupleElem : tuple.elems()) {
                if (! (tupleElem instanceof VarrefItree tupleElemVarref)) {
                    return null;
                }

                if (! (tupleElemVarref.owner() instanceof BindingItree)) {
                    return null;
                }

                lvars.add(new LocalVar.Original(tupleElemVarref.sym()));
            }
            return new NestedParam.Tuple(lvars);
        } else {
            return null;
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
