package org.kink_lang.kink.internal.program.itreeoptimize.ssafy;

import java.io.Serial;
import java.util.HashSet;
import java.util.Set;

import org.kink_lang.kink.internal.program.itree.*;

/**
 * Determines whether a fun can be converted to SSA.
 */
public class SsaChecker implements DeepTransformer.Callback {

    /**
     * Determines whether a fun can be converted to SSA.
     *
     * @param body the fun body to test.
     * @return whether the fun can be converted to SSA.
     */
    public static boolean canConvertToSsa(Itree body) {
        try {
            DeepTransformer.deepTransform(body, new SsaChecker());
            return true;
        } catch (CannotSsafyException ex) {
            return false;
        }
    }

    /** Syms which have been captured by funs. */
    private final Set<LocalVar> captured = new HashSet<>();

    /** Whether invocation has occurred. */
    private boolean hasBeenInvoked = false;

    /**
     * Capture the syms.
     */
    private void capture(Set<LocalVar> capturedLvars) {
        this.captured.addAll(capturedLvars);
    }

    /**
     * Does an invocation.
     */
    private void invocation() {
        this.hasBeenInvoked = true;
    }

    @Override
    public LocalVar derefLvar(LocalVar lvar) {
        return lvar;
    }

    @Override
    public LocalVar storeLvar(LocalVar lvar) {
        if (hasBeenInvoked || this.captured.contains(lvar)) {
            throw new CannotSsafyException();
        }
        return lvar;
    }

    @Override
    public Itree itree(Itree itree) {
        itree.accept(new Visitor());
        return itree;
    }

    /**
     * Visitor for the callback of DeepTransformer.
     */
    private class Visitor extends SkeltonItreeVisitor<Void> {

        /**
         * Constructs a visitor.
         */
        Visitor() {
            super(itree -> null);
        }

        @Override
        public Void visit(BindingItree itree) {
            throw new CannotSsafyException();
        }

        @Override
        public Void visit(SlowFunItree itree) {
            throw new CannotSsafyException();
        }

        @Override
        public Void visit(FastFunItree itree) {
            capture(itree.freeLvars());
            return null;
        }

        @Override
        public Void visit(AssignmentItree itree) {
            invocation();
            return null;
        }

        @Override
        public Void visit(BiArithmeticItree itree) {
            invocation();
            return null;
        }

        @Override
        public Void visit(NoTraitNewValItree itree) {
            invocation();
            return null;
        }

        @Override
        public Void visit(TraitNewValItree itree) {
            invocation();
            return null;
        }

        @Override
        public Void visit(IfItree itree) {
            invocation();
            return null;
        }

        @Override
        public Void visit(BranchItree itree) {
            invocation();
            return null;
        }

        @Override
        public Void visit(BranchWithElseItree itree) {
            invocation();
            return null;
        }

        @Override
        public Void visit(McallItree itree) {
            invocation();
            return null;
        }

        @Override
        public Void visit(SymcallItree itree) {
            invocation();
            return null;
        }

    }

    /**
     * Unchecked exception for an itree which cannot be converted to SSA.
     *
     * It inherits RuntimeException, because ItreeVisitor does not allow checked exceptions
     */
    public static class CannotSsafyException extends RuntimeException {

        /** Serial version UID of the exception class. */
        @Serial
        private static final long serialVersionUID = -5283922630086248066L;

    }

}

// vim: et sw=4 sts=4 fdm=marker
