package org.kink_lang.kink.internal.vec;

import javax.annotation.CheckReturnValue;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.SharedVars;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Immutable vec internal for elements
 * like ['A' a 'B' b 'C' c].
 */
public final class TraitVecInternal extends VecInternal implements MaybeTrait {

    /** The shared vars corresnponding to the elements. */
    private final SharedVars traitVars;

    /**
     * Constructs an immutable vec internal.
     */
    TraitVecInternal(Vm vm, Val[] vals, SharedVars traitVars) {
        super(vm, vals, vals.length);
        this.traitVars = traitVars;
    }

    /**
     * Returns the trait vars.
     *
     * @return the trait vars.
     */
    public SharedVars getTraitVars() {
        return this.traitVars;
    }

    @Override
    public TraitVecInternal getTrait() {
        return this;
    }

    @Override
    @CheckReturnValue
    public VecInternal set(int index, Val val) {
        Preconds.checkElemIndex(index, size());
        Val[] newVals = new Val[size()];
        copyTo(newVals, 0, size());
        newVals[index] = val;
        return new MutableVecInternal(vm, newVals, size());
    }

    @Override
    @CheckReturnValue
    public VecInternal remove(int index) {
        return removeRange(index, index + 1);
    }

    @Override
    @CheckReturnValue
    public VecInternal removeRange(int from, int to) {
        Preconds.checkRange(from, to, size());
        Val[] newVals = new Val[size() - (to - from)];
        copyOut(0, from, newVals, 0);
        copyOut(to, size() - to, newVals, from);
        return new MutableVecInternal(vm, newVals, size() - (to - from));
    }

    @Override
    @CheckReturnValue
    public VecInternal insert(int index, Val val) {
        Preconds.checkPosIndex(index, size());
        Val[] newVals = new Val[size() + 1];
        copyOut(0, index, newVals, 0);
        newVals[index] = val;
        copyOut(index, size() - index, newVals, index + 1);
        return new MutableVecInternal(vm, newVals, size() + 1);
    }

    @Override
    @CheckReturnValue
    public VecInternal insertAll(int index, VecInternal added) {
        Preconds.checkPosIndex(index, size());
        int addedSize = added.size();
        Val[] newVals = new Val[size() + addedSize];
        copyOut(0, index, newVals, 0);
        added.copyOut(0, addedSize, newVals, index);
        copyOut(index, size() - index, newVals, index + addedSize);
        return new MutableVecInternal(vm, newVals, size() + addedSize);
    }

    @Override
    @CheckReturnValue
    public VecInternal insertRange(int index, Val[] src, int from, int to) {
        Preconds.checkPosIndex(index, size());
        Preconds.checkRange(from, to, src.length);
        int addedSize = to - from;
        Val[] newVals = new Val[size() + addedSize];
        copyOut(0, index, newVals, 0);
        System.arraycopy(src, from, newVals, index, addedSize);
        copyOut(index, size() - index, newVals, index + addedSize);
        return new MutableVecInternal(vm, newVals, size() + addedSize);
    }

}

// vim: et sw=4 sts=4 fdm=marker
