package org.kink_lang.kink;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.StringJoiner;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * A <a href="../../../../../manual/api/kink-BIN.html#type-bin">
 * bin val</a>, which represents an immutable array of bytes.
 *
 * <p>A bin val holds Java byte arrays as a binary tree in order to avoid
 * allocation of the entire array on concatenation.
 * The byte array is substantiated on demand.</p>
 *
 * @see Vm#bin
 */
public class BinVal extends Val implements Comparable<BinVal> {

    /**
     * A node containing the bytes.
     *
     * Access to this field is not synchronized.
     * It is safe because:
     *
     * 1. BytesCell is immutable, and
     * 2. reading a stale value does not cause inconsistent result,
     *    although it may cause otherwise unneeded reinvocation of {@link BytesCell#singleCell()}.
     */
    private BytesCell cell;

    /**
     * Constructs a bin val, not copying the bytes.
     *
     * @param vm the vm.
     * @param bytes the source bytes.
     */
    BinVal(Vm vm, byte[] bytes) {
        this(vm, new BytesCell(bytes));
    }

    /**
     * Constructs a bin val.
     */
    private BinVal(Vm vm, BytesCell cell) {
        super(vm, null);
        Preconds.checkArg(cell.size() <= vm.bin.getMaxSize(),
                "size must not exceed getMaxSize()");
        this.cell = cell;
    }

    /**
     * Returns the size of the bin.
     *
     * @return the size of the bin.
     */
    public int size() {
        return this.cell.size();
    }

    /**
     * Returns the copied bytes of the bin.
     *
     * @return the copied bytes of the bin.
     */
    public byte[] bytes() {
        return Arrays.copyOf(rawBytes(), size());
    }

    /**
     * Copy the range of this bin into {@code bytes}
     * at the {@code at} index.
     *
     * @param from the inclusive start index.
     * @param to the exclusive end index.
     * @param bytes the destination bytes.
     * @param at the start index of the destination bytes.
     */
    public void copyToBytes(int from, int to, byte[] bytes, int at) {
        int length = to - from;
        System.arraycopy(rawBytes(), from, bytes, at, length);
    }

    /**
     * Makes a real byte array if not made,
     * and returns it.
     */
    private byte[] rawBytes() {
        BytesCell singleCell = this.cell.singleCell();
        this.cell = singleCell;
        return singleCell.back();
    }

    /**
     * Returns the byte at the ind.
     *
     * @param ind the index of the byte.
     * @return the byte at the ind.
     * @throws IndexOutOfBoundsException if {@code ind} is out of the range of the bin.
     */
    public byte get(int ind) {
        Preconds.checkElemIndex(ind, size());
        return rawBytes()[ind];
    }

    /**
     * Returns a read-only {@link ByteBuffer} of the content of this bin.
     *
     * <p>The capacity of the result ByteBuffer is the size of the bin.
     * The limit of the result ByteBuffer is set to the size of the bin.
     * The position of the result ByteBuffer is set to 0.</p>
     *
     * @return a read-only {@link ByteBuffer} of the content of this bin.
     */
    public ByteBuffer readOnlyByteBuffer() {
        ByteBuffer mutBuf = ByteBuffer.wrap(rawBytes());
        return mutBuf.asReadOnlyBuffer();
    }

    /**
     * Returns a sliced bin between {@code from} (inclusive) and {@code to} (exclusive).
     *
     * @param from the inclusive index where the result is sliced from.
     * @param to the exclusive index where the result is sliced to.
     * @return a sliced bin.
     * @throws IndexOutOfBoundsException
     *      if {@code from} or {@code to} are out of the range of the bin.
     * @throws IllegalArgumentException if {@code from} is larger than {@code to}.
     */
    public BinVal slice(int from, int to) {
        Preconds.checkRange(from, to, size());
        return new BinVal(vm, Arrays.copyOfRange(rawBytes(), from, to));
    }

    /**
     * Returns a negative number if {@code this} is less than {@code arg},
     * 0 if {@code this} is equivalent to {@code arg},
     * or a positive number if {@code this} is greater than {@code arg}.
     *
     * <p>Comparison is done for the bytes of {@code this} and {@code arg},
     * treating the bytes as unsigned numbers.</p>
     *
     * <p>Comparison is agnostic with {@code vm} field.</p>
     *
     * @param arg the arg bin to compare with.
     * @return the comparison number.
     */
    @Override
    public int compareTo(BinVal arg) {
        return Arrays.compareUnsigned(rawBytes(), arg.rawBytes());
    }

    /**
     * Returns a bin val which concatenates {@code this} and {@code tail}.
     *
     * @param tail the bin which is appended to the result.
     * @return a bin val which concatenates {@code this} and {@code tail}.
     */
    public BinVal concat(BinVal tail) {
        Preconds.checkArg((long) size() + tail.size() <= vm.bin.getMaxSize(),
                "result size exceeds the max");

        if (size() == 0) {
            return tail;
        } else if (tail.size() == 0) {
            return this;
        }

        return new BinVal(vm, this.cell.concat(tail.rawBytes()));
    }

    /**
     * Returns true if and only if {@code arg} is a bin val
     * and has the same vm and the equivalent bytes with {@code this}.
     *
     * @param arg the arg object to compare with.
     * @return true if and only if {@code arg} is a bin val
     *      and has the same vm and the equivalent bytes with {@code this}.
     */
    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof BinVal argBin
            && this.vm.equals(argBin.vm)
            && Arrays.equals(rawBytes(), argBin.rawBytes());
    }

    @Override
    public int hashCode() {
        return this.vm.hashCode() * 101
            + Arrays.hashCode(this.rawBytes());
    }

    @Override
    public String toString() {
        StringJoiner sj = new StringJoiner(" ", "BinVal(", ")");
        for (byte b : rawBytes()) {
            String s = Integer.toString(Byte.toUnsignedInt(b), 16);
            sj.add((s.length() == 1 ? "0x0" : "0x") + s);
        }
        return sj.toString();
    }

    @Override
    SharedVars sharedVars() {
        return vm.bin.sharedVars;
    }

    /**
     * Containing the bytes, as a reversed cons list.
     *
     * It is optimized for concatenating like (((x + y) + z) + w).
     *
     * @param size the size of the bytes.
     * @param front bin on the front side; may be null.
     * @param back bytes at the back side.
     */
    private record BytesCell(int size, @Nullable BytesCell front, byte[] back) {

        /**
         * Constructs a single cell.
         */
        BytesCell(byte[] back) {
            this(back.length, null, back);
        }

        /**
         * Returns a cell which is equivalent to {@code this}
         * and whose front() is null.
         */
        BytesCell singleCell() {
            if (front() == null) {
                return this;
            }

            byte[] bytes = new byte[size()];
            int end = size();
            BytesCell cell = this;
            while (cell != null) {
                int start = end - cell.back().length;
                System.arraycopy(cell.back(), 0, bytes, start, cell.back().length);

                end = start;
                cell = cell.front();
            }
            return new BytesCell(bytes);
        }

        /**
         * Make a cell concatenating {@code tail}.
         */
        BytesCell concat(byte[] tail) {
            return new BytesCell(size() + tail.length, this, tail);
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
