package org.kink_lang.kink;

import java.util.Arrays;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.contract.Preconds;
import org.kink_lang.kink.internal.vec.VecInternal;

/**
 * The dataStack of an execution stack.
 *
 * <p>Before calling methods which adds elements,
 * {@link #ensureCapa(int)} must be called
 * to allocate the capacity required for the make.</p>
 */
class DataStack {

    /** The vm. */
    private final Vm vm;

    /** Array containing the elements of the dataStack. After sp are null. */
    private Val[] array;

    /** The maximum capacity. */
    private final int maxCapa;

    /** The current stack top. */
    private int sp;

    /** The pointer to the base of the current call frame. */
    private int bp;

    /**
     * Constructs a dataStack.
     *
     * @param vm the vm.
     * @param initialCapa the initial capacity.
     * @param maxCapa the maximum capacity.
     */
    DataStack(Vm vm, int initialCapa, int maxCapa) {
        this.vm = vm;
        this.array = new Val[initialCapa];
        this.maxCapa = maxCapa;
    }

    /**
     * Returns the current stack top.
     */
    int sp() {
        return this.sp;
    }

    /**
     * Returns the pointer to the base of the current call frame.
     */
    int bp() {
        return this.bp;
    }

    /**
     * Moves the bp to right.
     */
    void increaseBp(int delta) {
        Preconds.checkPosIndex(delta, topOffset());
        this.bp += delta;
    }

    /**
     * Moves the bp to left.
     */
    void decreaseBp(int delta) {
        Preconds.checkPosIndex(delta, this.bp);
        this.bp -= delta;
    }

    /**
     * Returns the offset of the stack top from the bp.
     */
    int topOffset() {
        return this.sp - this.bp;
    }

    /**
     * Returns the backing array.
     */
    Val[] backingArray() {
        return this.array;
    }

    /**
     * Extends the capacity if necessary,
     * so that the dataStack has a capacity greater than or equal to {@code minCapa}.
     *
     * @return {@code true} if the required capacity is ensured.
     */
    private boolean ensureCapa(int minCapa) {
        if (this.array.length >= minCapa) {
            return true;
        }

        if (minCapa > this.maxCapa) {
            return false;
        }

        int newCapa = Math.min((int) (minCapa * 1.25), this.maxCapa);
        Val[] newArray = new Val[newCapa];
        System.arraycopy(this.array, 0, newArray, 0, this.sp);
        this.array = newArray;
        return true;
    }

    /**
     * Extends the capacity if necessary,
     * so that the dataStack has a capacity greater than or equal to
     * {@code this.sp + plus}.
     *
     * @return {@code true} if the required capacity is ensured.
     */
    boolean ensureCapaSpPlus(int plus) {
        return ensureCapa(this.sp + plus);
    }

    /**
     * Returns the val at the offset.
     */
    Val atOffset(int offset) {
        Preconds.checkElemIndex(offset, topOffset());
        return this.array[this.bp + offset];
    }

    /**
     * Sets the val at the offset.
     */
    void setAtOffset(int offset, Val val) {
        Preconds.checkElemIndex(offset, topOffset());
        this.array[this.bp + offset] = val;
    }

    /**
     * Pushes an element.
     */
    void push(Val val) {
        Preconds.checkState(this.sp < array.length, "not enough capa");
        this.array[this.sp] = val;
        ++ this.sp;
    }

    /**
     * Increases the sp.
     */
    void increaseSp(int count) {
        int newSp = this.sp + count;
        Preconds.checkArg(newSp <= array.length, "not enough capa");
        this.sp = newSp;
    }

    /**
     * Pops a val from the stack top.
     */
    Val pop() {
        Preconds.checkState(this.sp >= 1, "no val to pop");
        -- this.sp;
        Val popped = this.array[this.sp];
        this.array[this.sp] = null;
        return popped;
    }

    /**
     * Returns the recv of the current call frame.
     */
    Val recv() {
        return atOffset(0);
    }

    /**
     * Returns the arg of the current call frame.
     */
    @Nullable
    Val arg(int argIndex) {
        return atOffset(1 + argIndex);
    }

    /**
     * Returns the arg vec of the current call frame.
     */
    VecVal argVec(int argCount) {
        int argStart = this.bp + 1;
        int argEnd = argStart + argCount;
        return vm.vec.of(this.array, argStart, argEnd);
    }

    /**
     * Pushes all vals from the array.
     */
    void pushAll(Val[] vals) {
        Preconds.checkArg(this.sp + vals.length <= this.array.length, "not enough capa");
        System.arraycopy(vals, 0, this.array, this.sp, vals.length);
        this.sp += vals.length;
    }

    /**
     * Pushes the vals from [0] to [size - 1] of the vecInternal.
     */
    void pushAll(VecInternal vecInternal, int size) {
        Preconds.checkArg(this.sp + size <= this.array.length, "not enough capa");
        vecInternal.copyTo(this.array, this.sp, size);
        this.sp += size;
    }

    /**
     * Removes vals from the bp, to the bp + offset.
     */
    void removeToOffset(int offset) {
        int truncTo = this.bp + offset;
        Preconds.checkArg(truncTo <= this.sp, "too big offset");
        int moved = this.sp - truncTo;
        System.arraycopy(this.array, truncTo, this.array, this.bp, moved);
        int newSp = this.bp + moved;
        Arrays.fill(this.array, newSp, this.sp, null);
        this.sp = newSp;
    }

    /**
     * Removes vals from the bp + offset.
     */
    void removeFromOffset(int offset) {
        int from = this.bp + offset;
        Preconds.checkArg(from <= this.sp, "too big offset");
        Arrays.fill(this.array, from, this.sp, null);
        this.sp = from;
    }

    /**
     * Makes a vec from the top vals from the bp plus {@code offset}.
     */
    VecVal makeVecFromOffset(int offset) {
        return vm.vec.of(this.array, this.bp + offset, this.sp);
    }

    /**
     * Returns a slice of the vals.
     */
    Val[] slice(int from, int to) {
        Preconds.checkRange(from, to, this.sp);
        Val[] result = new Val[to - from];
        System.arraycopy(this.array, from, result, 0, to - from);
        return result;
    }

    /**
     * Returns a slice from the top.
     */
    Val[] sliceTop(int size) {
        return slice(this.sp - size, this.sp);
    }

}

// vim: et sw=4 sts=4 fdm=marker
