package org.kink_lang.kink;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Optional;

/**
 * An <a href="../../../../../manual/api/kink-EXCEPTION.html#type-exception">exception val</a>.
 *
 * @see Vm#exception
 */
public class ExceptionVal extends Val {

    /** The exception message. */
    private final String message;

    /** The traces. */
    private final List<TraceVal> traces;

    /** The exception which is next to this in the chain.. */
    private final Optional<ExceptionVal> next;

    /**
     * The one true constructor.
     */
    private ExceptionVal(
            Vm vm,
            String message,
            List<? extends TraceVal> traces,
            Optional<ExceptionVal> next) {
        super(vm);
        this.message = message;
        this.traces = List.copyOf(traces);
        this.next = next;
    }

    /**
     * Constructs an exception val without cahining.
     */
    ExceptionVal(Vm vm, String message, List<? extends TraceVal> traces) {
        this(vm, message, traces, Optional.empty());
    }

    /**
     * Constructs an exception val with the next in the chain.
     */
    ExceptionVal(Vm vm, String message, List<? extends TraceVal> traces, ExceptionVal next) {
        this(vm, message, traces, Optional.of(next));
    }

    /**
     * Returns the message.
     *
     * @return the message.
     */
    public String message() {
        return this.message;
    }

    /**
     * Returns the traces.
     *
     * @return the traces.
     */
    public List<TraceVal> traces() {
        return this.traces;
    }

    /**
     * Returns the exception which is next to this in the chain.
     *
     * @return the exception which is next to this in the chain.
     */
    public Optional<ExceptionVal> next() {
        return this.next;
    }

    /**
     * Concatenates the chain of {@code this} and the chain of {@code tail}.
     *
     * @param tail the exception of the chain which is will be the tail of the result.
     * @return an exception of the concatenated chain.
     */
    public ExceptionVal chain(ExceptionVal tail) {
        ExceptionVal chained = tail;
        for (ExceptionVal head : fromBackToFront()) {
            chained = new ExceptionVal(vm, head.message(), head.traces(), chained);
        }
        return chained;
    }

    /**
     * Flattens the chain from the back to the front.
     */
    private List<ExceptionVal> fromBackToFront() {
        List<ExceptionVal> flattened = new ArrayList<>();
        ExceptionVal exc = this;
        while (true) {
            flattened.add(exc);
            if (exc.next().isEmpty()) {
                break;
            }
            exc = exc.next().get();
        }
        Collections.reverse(flattened);
        return Collections.unmodifiableList(flattened);
    }

    /**
     * Convert this exception to Java RuntimeException.
     *
     * @return the converted RuntimeException.
     */
    public RuntimeException toRuntimeException() {
        var rte = convertToRte(this);
        var parent = this;
        while (parent.next().isPresent()) {
            var suppressed = parent.next().get();
            rte.addSuppressed(convertToRte(suppressed));
            parent = suppressed;
        }
        return rte;
    }

    /**
     * Convert a single exception to RTE.
     */
    private static RuntimeException convertToRte(ExceptionVal exc) {
        List<TraceVal> myTraces = new ArrayList<>(exc.traces());
        Collections.reverse(myTraces);
        List<StackTraceElement> stes = myTraces.stream()
                .map(TraceVal::toStackTraceElement)
                .toList();
        var rte = new RuntimeException(exc.message());
        rte.setStackTrace(stes.stream().toArray(StackTraceElement[]::new));
        return rte;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "ExceptionVal(%s %s)", message(), next());
    }

    @Override
    SharedVars sharedVars() {
        return vm.exception.sharedVars;
    }

    /**
     * Properties of the exception val.
     */
    private List<Object> properties() {
        return List.of(vm, message, traces, next);
    }

    @Override
    public int hashCode() {
        return properties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof ExceptionVal argExc
            && properties().equals(argExc.properties());
    }

}

// vim: et sw=4 sts=4 fdm=marker
