package org.kink_lang.kink;

import java.lang.invoke.CallSite;
import java.lang.invoke.ConstantCallSite;
import java.lang.invoke.MutableCallSite;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles.Lookup;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;

import org.kink_lang.kink.internal.ovis.OwnVarIndexes;

/**
 * Bootstrap method of var getting operation.
 */
final class GetVarBootstrapper {

    /** The maximum index of val fields in Val class. */
    private static final int MAX_VAL_FIELD_IND = 5;

    /** Logger for the class. */
    private static final System.Logger LOGGER
        = System.getLogger(GetVarBootstrapper.class.getName());

    /** When true, always fall back to fallbackMH. */
    static boolean alwaysFallback
        = Boolean.getBoolean(GetVarBootstrapper.class.getName() + ".alwaysFallback");

    static {
        LOGGER.log(System.Logger.Level.DEBUG, "alwaysFallback = {0}", alwaysFallback);
    }

    /**
     * Not instantiated.
     */
    private GetVarBootstrapper() {
    }

    /**
     * Bootstrap of var getting.
     * The type of the CallSite: (Val owner)Val
     */
    static CallSite bootstrapGetVar(
            Lookup caller,
            String name,
            MethodType type,
            int symHandle) throws Throwable {
        if (alwaysFallback) {
            return new ConstantCallSite(fallbackMH(symHandle));
        } else {
            // Why not using MutableCallSite(MethodType):
            // There appears to be no guaratee that setTarget happens-before linking of indy.
            // https://docs.oracle.com/javase/specs/jls/se8/html/jls-17.html#jls-17.4.5
            //
            // Thus, when using MutableCallSite(MethodType) without syncAll,
            // null target might be observable from other threads.
            //
            // Here, by using MutableCallSite(MethodHandle),
            // other threads at least can see the write of fallbackMH.
            MutableCallSite callSite = new MutableCallSite(fallbackMH(symHandle));

            callSite.setTarget(profileGetVarMH(symHandle, callSite));
            return callSite;
        }
    }

    /**
     * MethodHandle of profiling get-var.
     */
    static MethodHandle profileGetVarMH(
            int symHandle,
            MutableCallSite callSite) throws Throwable {
        var method = MethodHandles.lookup()
            .findStatic(GetVarBootstrapper.class, "profileGetVar",
                    MethodType.methodType(Val.class, Val.class, int.class, MutableCallSite.class));
        return MethodHandles.insertArguments(method, 1, symHandle, callSite);
    }

    /**
     * Do profiling and get var.
     */
    static Val profileGetVar(
            Val owner,
            int symHandle,
            MutableCallSite callSite) throws Throwable {
        var mh = profile(owner, symHandle);
        callSite.setTarget(mh);
        return (Val) mh.invoke(owner);
    }

    /**
     * Make a MethodHandle based on profiling.
     */
    private static MethodHandle profile(Val owner, int symHandle) throws Throwable {
        var ovis = owner.getOvis();
        int varInd = ovis.getIndex(symHandle);
        if (varInd >= 0) {
            return cacheOwnVarMH(symHandle, ovis, varInd);
        }

        SharedVars sv = owner.sharedVars();
        int svInd = sv.getIndex(symHandle);
        if (svInd >= 0) {
            return cacheSharedVarMH(symHandle, ovis, sv.getUniqueId(), svInd);
        }

        return fallbackMH(symHandle);
    }

    /**
     * MethodHandle with no caching.
     */
    static MethodHandle fallbackMH(int symHandle) throws Throwable {
        MethodHandle getVarMh = MethodHandles.lookup()
            .findVirtual(Val.class, "getVar", MethodType.methodType(Val.class, int.class));
        return MethodHandles.insertArguments(getVarMh, 1, symHandle);
    }

    /**
     * MethodHandle caching an own var.
     */
    static MethodHandle cacheOwnVarMH(
            int symHandle,
            OwnVarIndexes ovis,
            int varInd) throws Throwable {
        return varInd <= MAX_VAL_FIELD_IND
            ? cacheValFieldMH(symHandle, ovis, varInd)
            : cacheMvMH(symHandle, ovis, varInd);
    }

    /**
     * MethodHandle caching an own var of a val field.
     */
    private static MethodHandle cacheValFieldMH(
            int symHandle,
            OwnVarIndexes ovis,
            int varInd) throws Throwable {
        String methodName = "getPredictV" + varInd;
        MethodHandle method = MethodHandles.lookup()
            .findVirtual(
                    Val.class,
                    methodName,
                    MethodType.methodType(Val.class,
                        int.class,
                        OwnVarIndexes.class));
        return MethodHandles.insertArguments(method, 1, symHandle, ovis);
    }

    /**
     * MethodHandle caching an own var of val.moreVars.
     */
    private static MethodHandle cacheMvMH(
            int symHandle,
            OwnVarIndexes ovis,
            int varInd) throws Throwable {
        MethodHandle method = MethodHandles.lookup()
            .findVirtual(
                    Val.class,
                    "getPredictMv",
                    MethodType.methodType(Val.class,
                        int.class,
                        OwnVarIndexes.class,
                        int.class));
        return MethodHandles.insertArguments(method, 1, symHandle, ovis, varInd);
    }

    /**
     * MethodHandle caching a shared var.
     */
    static MethodHandle cacheSharedVarMH(
            int symHandle,
            OwnVarIndexes ovis,
            long svUniqueId,
            int svInd) throws Throwable {
        MethodHandle method = MethodHandles.lookup()
            .findVirtual(
                    Val.class,
                    "getPredictShared",
                    MethodType.methodType(Val.class,
                        int.class,
                        OwnVarIndexes.class,
                        long.class,
                        int.class));
        return MethodHandles.insertArguments(method, 1, symHandle, ovis, svUniqueId, svInd);
    }

}

// vim: et sw=4 sts=4 fdm=marker
