package org.kink_lang.kink;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import org.kink_lang.kink.internal.function.ThrowingFunction1;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * The helper for {@linkplain LocationVal location vals}.
 *
 * @see Vm#location
 */
public final class LocationHelper {

    /** The vm. */
    private final Vm vm;

    /** Shared vars of location vals. */
    SharedVars sharedVars;

    /**
     * Constructs the helper.
     */
    LocationHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Returns a location.
     *
     * @param programName the name of the program.
     * @param programText the text of the program.
     * @param charPos the char index of the pos of the loc.
     * @return a location.
     */
    public LocationVal of(String programName, String programText, int charPos) {
        return new LocationVal(vm, programName, programText, charPos);
    }

    /**
     * Initializes the helper.
     */
    void init() {
        Map<Integer, Val> vars = new HashMap<>();
        vars.put(vm.sym.handleFor("program_name"),
                unaryOp("Loc.program_name", loc -> vm.str.of(loc.programName())));
        vars.put(vm.sym.handleFor("program_text"),
                unaryOp("Loc.program_text", loc -> vm.str.of(loc.programText())));
        vars.put(vm.sym.handleFor("pos"), unaryOp("Loc.pos", loc -> vm.num.of(loc.runePos())));
        vars.put(vm.sym.handleFor("line_num"), unaryOp("Loc.line_num",
                    loc -> vm.num.of(loc.lineNum())));
        vars.put(vm.sym.handleFor("col_offset"), unaryOp("Loc.col_offset",
                    loc -> vm.num.of(loc.columnRuneOffset())));
        vars.put(vm.sym.handleFor("col_num"), unaryOp("Loc.col_num",
                    loc -> vm.num.of(loc.columnRuneOffset() + 1)));
        vars.put(vm.sym.handleFor("line"), unaryOp("Loc.line", loc -> vm.str.of(loc.line())));
        vars.put(vm.sym.handleFor("op_eq"), vm.fun.make("Loc.op_eq").take(1).action(
                    c -> vm.bool.of(c.recv().equals(c.arg(0)))));
        vars.put(vm.sym.handleFor("repr"), unaryOp("Loc.repr", this::reprMethod));
        vars.put(vm.sym.handleFor("indicator"), unaryOp("Loc.indicator",
                    loc -> vm.str.of(loc.indicator())));
        vars.put(vm.sym.handleFor("desc"), unaryOp("Loc.desc",
                    loc -> vm.str.of(loc.desc())));
        this.sharedVars = vm.sharedVars.of(vars);
    }

    // Loc.repr {{{1

    /**
     * Implementation of Loc.repr.
     */
    private HostResult reprMethod(LocationVal loc) {
        return loc.equals(of("", "", 0))
            ? vm.str.of("(location empty)")
            : vm.str.of(String.format(Locale.ROOT, "(location %s)", loc.desc()));
    }

    // }}}1

    /**
     * Makes an nullary method fun.
     */
    private FunVal unaryOp(String prefix, ThrowingFunction1<LocationVal, HostResult> action) {
        return vm.fun.make(prefix).take(0).action(c -> {
            Val recv = c.recv();
            if (! (recv instanceof LocationVal)) {
                return c.call(vm.graph.raiseFormat("{}: Loc must be location, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(recv)));
            }
            return action.apply((LocationVal) recv);
        });
    }

}

// vim: et sw=4 sts=4 fdm=marker
