package org.kink_lang.kink;

import java.util.List;
import java.util.Locale;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * A <a href="../../../../../manual/api/kink-LOCATION.html#type-location">location</a> in a program.
 *
 * @see Vm#location
 */
public class LocationVal extends Val {

    /** The program name. */
    private final String programName;

    /** The program text. */
    private final String programText;

    /** The char index of the pos. */
    private final int charPos;

    /**
     * Constructs a location.
     */
    LocationVal(Vm vm, String programName, String programText, int charPos) {
        super(vm, null);
        Preconds.checkPosIndex(charPos, programText.length());
        this.programName = programName;
        this.programText = programText;
        this.charPos = charPos;
    }

    /**
     * Returns the program name.
     *
     * @return the program name.
     */
    public String programName() {
        return this.programName;
    }

    /**
     * Returns the program text.
     *
     * @return the program text.
     */
    public String programText() {
        return this.programText;
    }

    /**
     * Returns the char index of the pos.
     *
     * @return the char index of the pos.
     */
    public int charPos() {
        return this.charPos;
    }

    /**
     * Returns the rune index of the pos.
     *
     * @return the rune index of the pos.
     */
    public int runePos() {
        return programText.codePointCount(0, charPos());
    }

    /**
     * Returns the one-based line number of the location on the program text.
     *
     * @return the one-based line number of the location on the program text.
     */
    public int lineNum() {
        int lineOffset = (int) programText().subSequence(0, charPos()).codePoints()
            .filter(rune -> rune == '\n')
            .count();
        return lineOffset + 1;
    }

    /**
     * Returns the char offset from the head of the line.
     *
     * @return the char offset from the head of the line.
     */
    public int columnCharOffset() {
        return charPos() - lineHeadCharIndex();
    }

    /**
     * Returns the rune offset from the head of the line.
     *
     * @return the rune offset from the head of the line.
     */
    public int columnRuneOffset() {
        return programText().codePointCount(lineHeadCharIndex(), charPos());
    }

    /**
     * Returns the line string where the loc is located.
     *
     * @return the line string where the loc is located.
     */
    public String line() {
        return programText().substring(lineHeadCharIndex(), lineTailCharIndex());
    }

    /**
     * Returns the string of {@code Trace.indicator}.
     *
     * The example result: "foo(--&gt;bar)".
     *
     * @return the string of {@code Trace.indicator}.
     */
    public String indicator() {
        String line = line();
        String before = line.substring(0, columnCharOffset());
        String after = line.substring(columnCharOffset());
        return String.format(Locale.ROOT, "%s-->%s", before, after).trim();
    }

    /**
     * Returns the string of {@code Trace.desc}.
     *
     * The example result: "foo.kn L10 C15".
     *
     * @return the string of {@code Trace.desc}.
     */
    public String desc() {
        return String.format(Locale.ROOT, "%s L%d C%d",
                programName(), lineNum(), columnRuneOffset() + 1);
    }

    /**
     * Returns the head index of the line.
     */
    private int lineHeadCharIndex() {
        int lfInd = programText().lastIndexOf('\n', charPos() - 1);
        return lfInd < 0 ? 0 : lfInd + 1;
    }

    /**
     * Returns the tail index of the line.
     */
    private int lineTailCharIndex() {
        int lfInd = programText().indexOf('\n', charPos());
        return lfInd < 0 ? programText().length() : lfInd + 1;
    }

    /**
     * Returns the string representation of the location val.
     *
     * @return the string representation of the location val.
     */
    @Override
    public String toString() {
        return String.format(Locale.ROOT,
                "LocationVal(%s L%d C%d)",
                programName, lineNum(), columnRuneOffset() + 1);
    }

    /**
     * Returns properties which determine equality of trace locations.
     */
    private List<Object> properties() {
        return List.of(vm, programName, programText, charPos);
    }

    /**
     * Returns the hash code of the loction val.
     *
     * @return the hash code of the location val.
     */
    @Override
    public int hashCode() {
        return properties().hashCode();
    }

    /**
     * Returns {@code true} if the two locations have the same vms, program names, program texts
     * and pos indices.
     *
     * @param arg the argument to compare to.
     * @return {@code true} if the two locations are equal.
     */
    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof LocationVal
            && properties().equals(((LocationVal) arg).properties());
    }

    @Override
    SharedVars sharedVars() {
        return vm.location.sharedVars;
    }

}

// vim: et sw=4 sts=4 fdm=marker
