package org.kink_lang.kink;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.function.Supplier;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.function.ThrowingFunction2;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * The registry of modules.
 */
class ModRegistry {

    /** The vm. */
    private final Vm vm;

    /** The map of loaded mods. */
    private final Map<String, Val> loadedMods = Collections.synchronizedMap(new HashMap<>());

    /** The map of mod producers. */
    private final Map<String, Supplier<Val>> modProducers
        = Collections.synchronizedMap(new HashMap<>());

    /**
     * Makes the registry.
     */
    ModRegistry(Vm vm) {
        this.vm = vm;
    }

    /**
     * Puts the val as the mod with the name.
     *
     * <p>If the registry already has a mod with the name,
     * the method does not register the mod.</p>
     *
     * @param modName the name of the mod.
     * @param mod the mod val.
     */
    void put(String modName, Val mod) {
        this.loadedMods.putIfAbsent(modName, mod);
    }

    /**
     * Returns the specified mod if already loaded, or null.
     */
    @Nullable
    Val getLoaded(String modName) {
        return this.loadedMods.get(modName);
    }

    /**
     * Registers a producer of a mod with the name.
     *
     * <p>When the mod name is requried for the first time,
     * the producer is invoked to produce the mod.</p>
     *
     * <p>Precondition: the producer with the mod name must not have been.</p>
     *
     * @param modName the mod name.
     * @param modProducer the producer of a mod.
     * @throws IllegalArgumentException if the producer with the name is already registered.
     */
    void register(String modName, Supplier<Val> modProducer) {
        Supplier<Val> alreadyRegistered = this.modProducers.putIfAbsent(modName, modProducer);
        if (alreadyRegistered != null) {
            throw new IllegalArgumentException(String.format(Locale.ROOT,
                        "mod producer already registered: %s", modName));
        }
    }

    /**
     * Implementation of require.
     */
    HostResult require(
            HostContext c,
            String modName,
            @Nullable FunVal successCont,
            @Nullable FunVal notFoundCont,
            @Nullable FunVal compileErrorCont) throws Throwable {
        Val loaded = loadedMods.get(modName);
        if (loaded != null) {
            return continueToSuccess(c, successCont, loaded);
        }

        Supplier<Val> producer = modProducers.get(modName);
        if (producer != null) {
            var mod = producer.get();
            mod.setVar(vm.sym.handleFor("repr"), modReprMethod(mod, modName));
            put(modName, mod);
            return continueToSuccess(c, successCont, loadedMods.get(modName));
        }

        if (! isGoodName(modName)) {
            return notFoundCont == null
                ? callDefaultNotFoundCont(c, modName)
                : c.call(notFoundCont);
        }

        HostResult builtin = loadBuiltin(c, modName,
                (cc, mod) -> continueToSuccess(cc, successCont, mod));
        if (builtin != null) {
            return builtin;
        }

        return loadFromFs(c, modName, successCont, notFoundCont, compileErrorCont);
    }

    /**
     * Calls the success cont, or falls back to the default action.
     */
    private HostResult continueToSuccess(HostContext c, @Nullable FunVal successCont, Val mod) {
        if (successCont == null) {
            return mod;
        }

        return c.call(successCont).args(mod);
    }

    /**
     * Loads the mod from the file system.
     */
    private HostResult loadFromFs(
            HostContext c,
            String modName,
            @Nullable FunVal successCont,
            @Nullable FunVal notFoundCont,
            @Nullable FunVal compileErrorCont
            ) throws Throwable {
        FunVal putAndSuccess = vm.fun.make().take(1).action(cc -> {
            Val mod = cc.arg(0);
            put(modName, mod);
            return continueToSuccess(cc, successCont, loadedMods.get(modName));
        });
        FunVal nonnullNotFoundCont = notFoundCont != null
            ? notFoundCont
            : vm.fun.make().take(0).action(cc -> callDefaultNotFoundCont(cc, modName));
        FunVal nonnullCompileErrorCont = compileErrorCont != null
            ? compileErrorCont
            : vm.fun.make().take(3).action(cc -> callDefaultCompileErrorCont(cc));
        int handle = vm.sym.handleFor("load_from_fs");
        return c.call("kink/_mod/LOAD_FS_MOD", handle).args(
                vm.str.of(modName),
                putAndSuccess,
                nonnullNotFoundCont,
                nonnullCompileErrorCont);
    }

    /**
     * Calls REQUIRE_AUX.default_not_found_cont.
     */
    private HostResult callDefaultNotFoundCont(HostContext c, String modName) {
        int handle = vm.sym.handleFor("default_not_found_cont");
        return c.call("kink/_mod/REQUIRE_AUX", handle).args(vm.str.of(modName));
    }

    /**
     * Calls REQUIRE_AUX.default_compile_error_cont.
     */
    private HostResult callDefaultCompileErrorCont(CallContext c) {
        int handle = vm.sym.handleFor("default_compile_error_cont");
        Val successCont = c.arg(0);
        Val notFoundCont = c.arg(1);
        Val compileErrorCont = c.arg(2);
        return c.call("kink/_mod/REQUIRE_AUX", handle)
            .args(successCont, notFoundCont, compileErrorCont);
    }

    /**
     * Loads the mod from the Kink Java module.
     */
    @Nullable
    private HostResult loadBuiltin(
            HostContext c,
            String modName,
            ThrowingFunction2<HostContext, Val, HostResult> onFound
            ) throws Throwable {
        String fileName = toFileName(modName);
        Module javaMod = Vm.class.getModule();
        @SuppressWarnings("PMD.CloseResource")
        InputStream stream = javaMod.getResourceAsStream(fileName);
        if (stream == null) {
            return null;
        }

        byte[] bytes;
        try (stream) {
            bytes = stream.readAllBytes();
        }
        String programName = "builtin:" + fileName;
        String programText = new String(bytes, StandardCharsets.UTF_8);
        BindingVal mod = vm.binding.newBinding();
        FunVal fun = vm.fun.compile(
                Locale.ROOT, programName, programText,
                mod,
                f -> f,
                error -> {
                    String excMsg = String.format(Locale.ROOT, "compile error: %s [%s] %s",
                            error.message(),
                            error.from().desc(),
                            error.from().indicator());
                    throw new IllegalStateException(excMsg);
                });
        mod.setVar(vm.sym.handleFor("repr"), modReprMethod(mod, modName));
        return c.call(fun).on((cc, ignored) -> {
            put(modName, mod);
            return onFound.apply(cc, loadedMods.get(modName));
        });
    }

    /**
     * Implementation of Kink_mod.repr.
     */
    private FunVal modReprMethod(Val mod, String modName) {
        String reprMethodName = String.format(Locale.ROOT, "%s.repr", modName);
        return vm.fun.make(reprMethodName).take(0).action(c -> {
            Val recv = c.recv();
            return recv.equals(mod)
                ? vm.str.of(String.format(Locale.ROOT, "(mod %s)", modName))
                : vm.str.of(String.format(Locale.ROOT,
                            "(binding mod=%s val_id=%s)", modName, recv.identity()));
        });
    }

    /** The pattern of good module names. */
    private static final Pattern GOOD_NAME_PAT
        = Pattern.compile("([a-z_][a-z0-9_]*/)*_*[A-Z][A-Z0-9_]*");

    /**
     * Returns true if the modName is a good name.
     */
    boolean isGoodName(String modName) {
        return GOOD_NAME_PAT.matcher(modName).matches();
    }

    /**
     * Converts the module name to the file name.
     */
    private String toFileName(String modName) {
        return String.format(Locale.ROOT, "kink-mods/%s.kn", modName);
    }

}

// vim: et sw=4 sts=4 fdm=marker
